const { getBscTokenDetails } = require('./bsc');
const { getEthTokenDetails } = require('./ethereum');
const { getSolanaTokenDetails } = require('./solana');
const { fetchDexScreenerData } = require('../utils/dexScreener');
const { formatTokenAnalysis } = require('../utils/formatter');
const { getLpLockInfo } = require('../utils/lockInfoFetcher');
// Add missing import for analyzeTokenSecurity
const { analyzeTokenSecurity } = require('./securityAnalyzer');

/**
 * Main function to analyze a token
 */
async function analyzeToken(address, network) {
  const { saveScannedToken } = require('../utils/firstCallDb');
  // Accept groupId as optional third argument
  async function innerAnalyze(address, network, groupId = null, usernameCalled = null) {
    let tokenDetails;
    let dexData;
    // Ensure address and network are properly set
    if (!address) {
      throw new Error('Token address is required');
    }
    // If network is not explicitly provided, try to detect it from address format
    if (!network) {
      network = detectNetwork(address);
      if (!network) {
        throw new Error('Could not detect network from address');
      }
      console.log(`Detected network: ${network}`);
    }
    // ...existing code...
    
    // Special handling for Solana addresses
    if (network === 'SOL' || (address && !address.startsWith('0x'))) {
      console.log('Processing Solana token');
      network = 'SOL'; // Ensure consistent network name
      tokenDetails = await getSolanaTokenDetails(address);
      dexData = await fetchDexScreenerData(address, 'solana');
    }
    // Improved network detection for EVM chains
    else if (address.startsWith('0x')) {
      if (network === 'ETH') {
        try {
          console.log('Attempting auto-detection between BSC and ETH...');
          try {
            console.log('Trying ETH first...');
            tokenDetails = await getEthTokenDetails(address);
            if (tokenDetails && tokenDetails.name) {
              console.log('✅ Successfully confirmed ETH network');
              network = 'ETH';
            }
          } catch (ethError) {
            console.log('Not an ETH token or ETH RPC error, trying BSC...');
            try {
              tokenDetails = await getBscTokenDetails(address);
              if (tokenDetails && tokenDetails.name) {
                console.log('✅ Successfully detected BSC network');
                network = 'BSC';
              } else {
                console.log('BSC detection failed, defaulting to ETH');
                network = 'ETH';
              }
            } catch (bscError) {
              console.error('BSC detection error:', bscError.message);
              network = 'ETH';
            }
          }
        } catch (error) {
          console.error('Network detection error:', error.message);
        }
      }
      console.log(`🔍 Analyzing token on ${network} network...`);
      switch (network) {
        case 'BSC':
          tokenDetails = await getBscTokenDetails(address);
          dexData = await fetchDexScreenerData(address, 'bsc');
          break;
        case 'ETH':
          tokenDetails = await getEthTokenDetails(address);
          dexData = await fetchDexScreenerData(address, 'ethereum');
          break;
        default:
          throw new Error(`Unsupported EVM network: ${network}`);
      }
    } else {
      throw new Error(`Unsupported address format or network: ${network}`);
    }

  // Fetch market cap from dexData or tokenDetails. Prefer numeric dexData.marketCap if available.
    let mcap = null;
    const { parseMarketCap } = require('../utils/numberUtils');
    if (dexData) {
      // Try parsing dexData.marketCap first, fallback to dexData.mcap
      const parsed = parseMarketCap(dexData.marketCap !== undefined && dexData.marketCap !== null ? dexData.marketCap : dexData.mcap);
      if (parsed !== null && !isNaN(parsed)) {
        mcap = parsed;
      }
    }
    if (!mcap && typeof tokenDetails?.marketCap === 'number' && !isNaN(tokenDetails.marketCap)) {
      mcap = tokenDetails.marketCap;
    }
    if (!mcap && dexData && dexData.price && tokenDetails && tokenDetails.totalSupply) {
      // Fallback: hitung dari price * supply
      const price = parseFloat(dexData.price);
      const supply = parseFloat(tokenDetails.totalSupply);
      if (!isNaN(price) && !isNaN(supply)) {
        mcap = price * supply;
      }
    }
    if (!mcap) mcap = 'Unknown';
  // Save every scanned token to DB (only if market cap is valid)
    try {
      if (mcap !== 'Unknown') {
        // Use explicit usernameCalled parameter, fallback to globalThis if present (compat)
        let groupLink = null;
        if (typeof globalThis._tg_groupLink !== 'undefined') groupLink = globalThis._tg_groupLink;
        let usernameToSave = usernameCalled;
        if ((!usernameToSave || usernameToSave === 'Unknown') && typeof globalThis._tg_usernameCalled !== 'undefined') {
          usernameToSave = globalThis._tg_usernameCalled;
        }
        await saveScannedToken(address, network, groupId, mcap, groupLink, usernameToSave);
      }
    } catch (err) {
      console.error('Failed to save scanned token to DB:', err);
    }
    
    // Ensure correct network is passed through to the results
    if (tokenDetails) {
      tokenDetails.network = network;
    }

    // Do not display DexScreener URL by default
    if (dexData && dexData.url) {
      // Uncomment the line below to include the DexScreener URL in output
      // tokenDetails.dexScreenerUrl = dexData.url;
    }

    // Add social media links
    if (dexData && dexData.socials) {
      tokenDetails.socials = dexData.socials;
      console.log('Social media links found:', dexData.socials);
    }

    // Prefer owner data from DexScreener when available
    if (dexData && dexData.ownerInfo && tokenDetails.owner === 'Unknown') {
      console.log(`Using owner data from DexScreener: ${dexData.ownerInfo}`);
      tokenDetails.owner = dexData.ownerInfo;
    }

    // Prefer LP lock data from DexScreener when available
    if (dexData && dexData.lpLockStatus && dexData.lpLockStatus !== 'Unknown') {
      console.log(`Using LP Lock data from DexScreener: ${dexData.lpLockStatus}`);
      
      if (!tokenDetails.lpInfo) {
        tokenDetails.lpInfo = {
          dex: dexData.dexId || 'Unknown',
          locked: dexData.lpLockStatus
        };
      } else {
        tokenDetails.lpInfo.locked = dexData.lpLockStatus;
      }
      
      // Add more detailed lock info if available from DexScreener
      if (dexData.lpLockDetails) {
        console.log('Adding detailed lock information from DexScreener');
        if (!tokenDetails.lpInfo.platform && dexData.lpLockDetails.locker) {
          tokenDetails.lpInfo.platform = dexData.lpLockDetails.locker;
        }
        
        if (!tokenDetails.lpInfo.until && dexData.lpLockDetails.unlockDate) {
          tokenDetails.lpInfo.until = dexData.lpLockDetails.unlockDate;
        }
        
        if (!tokenDetails.lpInfo.lockLink && dexData.lpLockDetails.lockUrl) {
          tokenDetails.lpInfo.lockLink = dexData.lpLockDetails.lockUrl;
        }
      }
    }

    // Remove LP Lock info fetching since we're not using it anymore
    // Just use whatever comes directly from DexScreener
    if (!tokenDetails.lpInfo) {
      tokenDetails.lpInfo = {
        dex: dexData?.dexId || 'Unknown',
        locked: 'Unknown',
        platform: null,
        until: null,
        lockLink: null
      };
    }
    
    // Log data for debugging
    if (dexData) {
      console.log(`DexScreener data for ${address}:`);
      console.log(`- Market Cap: ${dexData.marketCap}`);
      console.log(`- Price: ${dexData.price}`);
      console.log(`- Liquidity: ${dexData.liquidityUSD}`);
      console.log(`- Volume 24h: ${dexData.volume24h}`);
      console.log(`- LP Lock: ${dexData.lpLockStatus || 'Unknown'}`);
      console.log(`- Verified: ${dexData.isVerified ? 'Yes' : 'No'}`);
    } else {
      console.log(`No DexScreener data found for ${address}`);
    }
    
    // Compute market cap if not available
    if (dexData && dexData.price && !dexData.marketCap) {
      // Only compute if there is truly no data from DexScreener
      console.log('No market cap from DexScreener, calculating from price and supply...');
      const price = parseFloat(dexData.price);
      const supply = parseFloat(tokenDetails.totalSupply);
      
      if (!isNaN(price) && !isNaN(supply)) {
        dexData.marketCap = price * supply;
        console.log(`Calculated market cap: ${dexData.marketCap}`);
      }
    }
    
    // Format and return the analysis with image
    const securityAnalysis = await analyzeTokenSecurity(tokenDetails, dexData);
    const analysis = await formatTokenAnalysis(tokenDetails, dexData, network);
    
    // Add security status to the result for image selection
    analysis.securityStatus = securityAnalysis.status;
    analysis.contractAddress = address; // Make sure contract address is available
    
    // Add network to the analysis result
    analysis.network = network;
    // Ensure symbol is available to callers (prefer tokenDetails, then DexScreener data)
    try {
      const dexSymbol = dexData && (dexData.symbol || (dexData.extendedDexData && dexData.extendedDexData.symbol));
      analysis.symbol = tokenDetails && tokenDetails.symbol ? tokenDetails.symbol : (dexSymbol || tokenDetails.name || 'UNKNOWN');
      if (typeof analysis.symbol === 'string') analysis.symbol = analysis.symbol.toUpperCase();
    } catch (e) {
      analysis.symbol = tokenDetails && tokenDetails.symbol ? tokenDetails.symbol : 'UNKNOWN';
    }
  // Include normalized market cap fields for callers to consume (numeric when possible)
  analysis.marketCap = (typeof mcap === 'number' && !isNaN(mcap)) ? mcap : (mcap || 'Unknown');
  analysis.mcap = dexData?.mcap ?? null;
  analysis.rawMcap = dexData?.mcap ?? null;
  analysis.fdv = dexData?.fdv ?? null;
    
    return analysis;
  }
  try {
    // If called with extra args, pass them to innerAnalyze: arguments[2]=groupId, arguments[3]=usernameCalled
    if (arguments.length > 2) {
      const groupIdArg = arguments[2];
      const usernameArg = arguments.length > 3 ? arguments[3] : null;
      return await innerAnalyze(address, network, groupIdArg, usernameArg);
    } else {
      return await innerAnalyze(address, network);
    }
  } catch (error) {
    console.error(`Error in analyzeToken:`, error);
    throw new Error(`Failed to analyze token: ${error.message}`);
  }
}

module.exports = {
  analyzeToken
};
