require('dotenv').config();
const { Web3 } = require('web3');  // Updated import style
const axios = require('axios');
const moment = require('moment');

const BASE_RPC = process.env.BASE_RPC || 'https://mainnet.base.org';
const web3 = new Web3(BASE_RPC);

// Standard ERC20 ABI (same as other EVM chains)
const ERC20_ABI = [
  { "inputs": [], "name": "name", "outputs": [{ "type": "string" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "symbol", "outputs": [{ "type": "string" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "decimals", "outputs": [{ "type": "uint8" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "totalSupply", "outputs": [{ "type": "uint256" }], "stateMutability": "view", "type": "function" },
  { "inputs": [{ "type": "address" }], "name": "balanceOf", "outputs": [{ "type": "uint256" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "owner", "outputs": [{ "type": "address" }], "stateMutability": "view", "type": "function" }
];

/**
 * Get Base token details
 */
async function getBaseTokenDetails(address) {
  try {
    const contract = new web3.eth.Contract(ERC20_ABI, address);
    
    // Basic token info
    const [name, symbol, decimalsStr, totalSupplyWei] = await Promise.all([
      contract.methods.name().call(),
      contract.methods.symbol().call(),
      contract.methods.decimals().call(),
      contract.methods.totalSupply().call()
    ]);
    
    const decimals = parseInt(decimalsStr);
    const totalSupply = Number(totalSupplyWei) / Math.pow(10, decimals);
    
    // Get owner address (may fail if not implemented)
    let owner = 'Unknown';
    try {
      owner = await contract.methods.owner().call();
      
      // Check if owner is null address (renounced)
      if (owner === '0x0000000000000000000000000000000000000000') {
        owner = 'RENOUNCED';
      }
    } catch (error) {
      console.log('Owner method not available');
    }
    
    // Get token contract creation date (simplified for Base)
    const creationInfo = {
      timestamp: Math.floor(Date.now() / 1000) - 86400 // Default to 1 day ago
    };
    
    // Get top holders (simplified for Base)
    const holders = {
      count: 0,
      top: []
    };
    
    // Check LP info - Base uses various DEXes
    const lpInfo = {
      dex: 'BaseSwap',
      lpAddress: 'Unknown',
      liquidity: 'Unknown',
      locked: 'Unknown'
    };
    
    // Get token age
    const age = moment().diff(moment.unix(creationInfo.timestamp), 'seconds');
    const ageFormatted = formatAge(age);
    
    // Get tax information (buy/sell)
    const taxInfo = {
      buy: '0.0%',
      sell: '0.0%'
    };
    
    return {
      name,
      symbol,
      decimals,
      totalSupply,
      owner,
      contractAddress: address,
      creationDate: creationInfo.timestamp,
      ageFormatted,
      holders,
      lpInfo,
      taxInfo
    };
  } catch (error) {
    console.error('Error getting Base token details:', error);
    throw new Error('Failed to get Base token details');
  }
}

/**
 * Format age to human-readable string
 */
function formatAge(seconds) {
  const days = Math.floor(seconds / 86400);
  const hours = Math.floor((seconds % 86400) / 3600);
  const minutes = Math.floor((seconds % 3600) / 60);
  
  return `${days}d - ${hours}h ${minutes}m ${seconds % 60}s`;
}

module.exports = {
  getBaseTokenDetails
};
