require('dotenv').config();
const { Web3 } = require('web3');  // Updated import style
const axios = require('axios');
const moment = require('moment');

// Standard ERC20 ABI for basic token interactions
const ERC20_ABI = [
  { "inputs": [], "name": "name", "outputs": [{ "type": "string" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "symbol", "outputs": [{ "type": "string" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "decimals", "outputs": [{ "type": "uint8" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "totalSupply", "outputs": [{ "type": "uint256" }], "stateMutability": "view", "type": "function" },
  { "inputs": [{ "type": "address" }], "name": "balanceOf", "outputs": [{ "type": "uint256" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "owner", "outputs": [{ "type": "address" }], "stateMutability": "view", "type": "function" }
];

const BSC_RPC = process.env.BSC_RPC || 'https://bsc-dataseed1.binance.org/';
// Add multiple RPC backup endpoints to handle connection issues
const BSC_RPC_BACKUPS = [
  'https://bsc-dataseed2.binance.org/',
  'https://bsc-dataseed3.binance.org/',
  'https://bsc-dataseed4.binance.org/',
  'https://bsc-dataseed.binance.org/',
  'https://binance.nodereal.io'
];
const BSC_API_KEY = process.env.BSC_API_KEY;
const BSCSCAN_API = `https://api.bscscan.com/api`;
let web3 = new Web3(BSC_RPC);

/**
 * Retry web3 requests with fallback RPC endpoints
 */
async function retryWithFallbacks(fn) {
  let lastError;
  
  try {
    return await fn(); // Try main endpoint first
  } catch (error) {
    console.log(`Error with primary BSC RPC, trying fallbacks: ${error.message}`);
    lastError = error;
  }
  
  // Try backup endpoints if main fails
  for (const rpcUrl of BSC_RPC_BACKUPS) {
    try {
      web3 = new Web3(rpcUrl);
      console.log(`Trying fallback RPC: ${rpcUrl}`);
      return await fn();
    } catch (error) {
      console.log(`Error with fallback RPC ${rpcUrl}: ${error.message}`);
      lastError = error;
    }
  }
  
  // If all fallbacks fail, throw the last error
  throw lastError;
}

/**
 * Get token details from BSC
 */
async function getBscTokenDetails(address) {
  try {
    console.log(`Getting BSC token details for ${address}`);
    
    return await retryWithFallbacks(async () => {
      const contract = new web3.eth.Contract(ERC20_ABI, address);
      
      // Basic token info with better error handling
      let name = 'Unknown';
      let symbol = 'Unknown';
      let decimals = 18;
      let totalSupply = 0;
      
      try {
        name = await contract.methods.name().call();
      } catch (e) {
        console.log('Error getting token name:', e.message);
      }
      
      try {
        symbol = await contract.methods.symbol().call();
      } catch (e) {
        console.log('Error getting token symbol:', e.message);
      }
      
      try {
        decimals = parseInt(await contract.methods.decimals().call());
      } catch (e) {
        console.log('Error getting token decimals:', e.message);
      }
      
      try {
        const totalSupplyWei = await contract.methods.totalSupply().call();
        totalSupply = Number(totalSupplyWei) / Math.pow(10, decimals);
      } catch (e) {
        console.log('Error getting token supply:', e.message);
      }
      
      // Enhanced owner detection - try multiple methods
      let owner = 'Unknown';
      
      // Method 1: Try to call owner() function on contract
      try {
        owner = await contract.methods.owner().call();
        
        // Check if owner is null address (renounced)
        if (owner === '0x0000000000000000000000000000000000000000') {
          owner = 'RENOUNCED';
        }
        console.log(`Owner from contract method: ${owner}`);
      } catch (error) {
        console.log('Owner method not available on contract');
      }
      
      // Method 2: If owner is still unknown, try to get contract creator from BscScan
      if (owner === 'Unknown') {
        try {
          console.log('Attempting to get contract creator from BscScan...');
          const response = await axios.get(BSCSCAN_API, {
            params: {
              module: 'contract',
              action: 'getcontractcreation',
              contractaddresses: address,
              apikey: BSC_API_KEY
            }
          });
          
          if (response.data.status === '1' && response.data.result && response.data.result[0]) {
            owner = response.data.result[0].contractCreator;
            console.log(`Owner from contract creation: ${owner}`);
          }
        } catch (error) {
          console.error('Error getting contract creator from BscScan:', error.message);
        }
      }
      
      // Method 3: Try to check for ownership transfer events using BscScan API
      if (owner !== 'RENOUNCED' && owner !== 'Unknown') {
        try {
          console.log('Checking ownership transfer events...');
          const response = await axios.get(BSCSCAN_API, {
            params: {
              module: 'account',
              action: 'txlist',
              address: address,
              startblock: 0,
              endblock: 999999999,
              sort: 'desc',
              apikey: BSC_API_KEY
            }
          });
          
          if (response.data.status === '1') {
            // Find ownership transfer events
            const transferEvents = response.data.result.filter(tx => 
              tx.functionName.includes('transferOwnership') ||
              tx.functionName.includes('renounceOwnership')
            );
            
            if (transferEvents.length > 0) {
              // Check the most recent ownership transfer
              const latestTransfer = transferEvents[0];
              if (latestTransfer.functionName.includes('renounceOwnership')) {
                owner = 'RENOUNCED';
                console.log('Ownership renounced according to transaction events');
              } else {
                // Extract the new owner from the input data
                // This is a simplified approach and might need additional parsing
                const inputData = latestTransfer.input;
                if (inputData && inputData.length >= 74) {
                  // Extract the address from the input data (typical format for transferOwnership)
                  const newOwner = '0x' + inputData.substring(34, 74);
                  if (web3.utils.isAddress(newOwner)) {
                    owner = newOwner;
                    console.log(`Latest owner from transfer events: ${owner}`);
                  }
                }
              }
            }
          }
        } catch (error) {
          console.error('Error checking ownership transfer events:', error.message);
        }
      }
      
      // Get token contract creation date
      const creationInfo = await getContractCreationInfo(address);
      
      // If owner is still Unknown, use the contract creator as owner
      if (owner === 'Unknown' && creationInfo && creationInfo.creator) {
        owner = creationInfo.creator;
        console.log(`Using contract creator as owner: ${owner}`);
      }
      
      // Get token age
      const age = moment().diff(moment.unix(creationInfo.timestamp), 'seconds');
      const ageFormatted = formatAge(age);
      
      // Get tax information (buy/sell)
      const taxInfo = await getTaxInfo(address);
      
      // Get LP information - THIS WAS MISSING
      const lpInfo = await getLPInfo(address);
      
      // Check if the contract is verified on BSCScan
      let verified = false;
      try {
        const verificationResponse = await axios.get(BSCSCAN_API, {
          params: {
            module: 'contract',
            action: 'getabi',
            address: address,
            apikey: BSC_API_KEY
          }
        });
        
        // If the ABI is available, the contract is verified
        verified = verificationResponse.data.status === '1';
        console.log(`Contract verification status: ${verified ? 'Verified' : 'Not verified'}`);
      } catch (error) {
        console.error('Error checking contract verification:', error.message);
      }
      
      return {
        name,
        symbol,
        decimals,
        totalSupply,
        owner,
        contractAddress: address,
        creationDate: creationInfo.timestamp,
        ageFormatted,
        lpInfo,
        taxInfo,
        verified // Add verification status
      };
    });
  } catch (error) {
    console.error('Error getting BSC token details:', error);
    throw new Error('Failed to get BSC token details: ' + error.message);
  }
}

/**
 * Get contract creation information
 */
async function getContractCreationInfo(address) {
  try {
    const response = await axios.get(BSCSCAN_API, {
      params: {
        module: 'contract',
        action: 'getcontractcreation',
        contractaddresses: address,
        apikey: BSC_API_KEY
      }
    });
    
    if (response.data.status === '1') {
      // Get block timestamp for creation date
      const txHash = response.data.result[0].txHash;
      const txInfo = await web3.eth.getTransaction(txHash);
      const block = await web3.eth.getBlock(txInfo.blockNumber);
      
      return {
        creator: response.data.result[0].contractCreator,
        txHash,
        timestamp: Number(block.timestamp) // Convert BigInt to Number
      };
    }
    
    return { timestamp: Math.floor(Date.now() / 1000) - 86400 }; // Default to 1 day ago
  } catch (error) {
    console.error('Error getting contract creation info:', error);
    return { timestamp: Math.floor(Date.now() / 1000) - 86400 }; // Default to 1 day ago
  }
}

/**
 * Get top token holders - function removed/disabled
 */
async function getTopHolders(address) {
  // Return empty data since holder functionality is removed
  return { count: 0, top: [] };
}

/**
 * Get LP information
 */
async function getLPInfo(address) {
  // This would normally involve checking PancakeSwap or other DEXes
  // Simplified implementation for now
  return {
    dex: 'PancakeSwap V3',
    lpAddress: '0xD770176e9102a4A61f1820b211ED1346305DE6D2', // Example
    liquidity: 'Unknown',
    locked: 'Unknown'
  };
}

/**
 * Get tax information
 */
async function getTaxInfo(address) {
  // This would require analyzing the token contract for transfer fees
  // Simplified implementation for now
  return {
    buy: '0.0%',
    sell: '0.0%'
  };
}

/**
 * Format age to human-readable string
 */
function formatAge(seconds) {
  const days = Math.floor(seconds / 86400);
  const hours = Math.floor((seconds % 86400) / 3600);
  const minutes = Math.floor((seconds % 3600) / 60);
  
  return `${days}d - ${hours}h ${minutes}m ${seconds % 60}s`;
}

module.exports = {
  getBscTokenDetails
};
