require('dotenv').config();
const { Web3 } = require('web3');  // Updated import style
const axios = require('axios');
const moment = require('moment');

const ETH_RPC = process.env.ETH_RPC;
const ETH_API_KEY = process.env.ETH_API_KEY;
const ETHERSCAN_API = 'https://api.etherscan.io/api';
const web3 = new Web3(ETH_RPC);

// Standard ERC20 ABI (same as BSC)
const ERC20_ABI = [
  { "inputs": [], "name": "name", "outputs": [{ "type": "string" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "symbol", "outputs": [{ "type": "string" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "decimals", "outputs": [{ "type": "uint8" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "totalSupply", "outputs": [{ "type": "uint256" }], "stateMutability": "view", "type": "function" },
  { "inputs": [{ "type": "address" }], "name": "balanceOf", "outputs": [{ "type": "uint256" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "owner", "outputs": [{ "type": "address" }], "stateMutability": "view", "type": "function" }
];

async function getEthTokenDetails(address) {
  try {
    const contract = new web3.eth.Contract(ERC20_ABI, address);
    
    // Basic token info
    const [name, symbol, decimalsStr, totalSupplyWei] = await Promise.all([
      contract.methods.name().call(),
      contract.methods.symbol().call(),
      contract.methods.decimals().call(),
      contract.methods.totalSupply().call()
    ]);
    
    const decimals = parseInt(decimalsStr);
    const totalSupply = Number(totalSupplyWei) / Math.pow(10, decimals);
    
    // Enhanced owner detection - try multiple methods
    let owner = 'Unknown';
    
    // Method 1: Try to call owner() function on contract
    try {
      owner = await contract.methods.owner().call();
      
      // Check if owner is null address (renounced)
      if (owner === '0x0000000000000000000000000000000000000000') {
        owner = 'RENOUNCED';
      }
      console.log(`Owner from contract method: ${owner}`);
    } catch (error) {
      console.log('Owner method not available on contract');
    }
    
    // Method 2: If owner is still unknown, try to get contract creator from Etherscan
    if (owner === 'Unknown') {
      try {
        console.log('Attempting to get contract creator from Etherscan...');
        const response = await axios.get(ETHERSCAN_API, {
          params: {
            module: 'contract',
            action: 'getcontractcreation',
            contractaddresses: address,
            apikey: ETH_API_KEY
          }
        });
        
        if (response.data.status === '1' && response.data.result && response.data.result[0]) {
          owner = response.data.result[0].contractCreator;
          console.log(`Owner from contract creation: ${owner}`);
        }
      } catch (error) {
        console.error('Error getting contract creator from Etherscan:', error.message);
      }
    }
    
    // Method 3: Try to check for ownership transfer events using Etherscan API
    if (owner !== 'RENOUNCED' && owner !== 'Unknown') {
      try {
        console.log('Checking ownership transfer events...');
        const response = await axios.get(ETHERSCAN_API, {
          params: {
            module: 'account',
            action: 'txlist',
            address: address,
            startblock: 0,
            endblock: 999999999,
            sort: 'desc',
            apikey: ETH_API_KEY
          }
        });
        
        if (response.data.status === '1') {
          // Find ownership transfer events
          const transferEvents = response.data.result.filter(tx => 
            tx.functionName.includes('transferOwnership') ||
            tx.functionName.includes('renounceOwnership')
          );
          
          if (transferEvents.length > 0) {
            // Check the most recent ownership transfer
            const latestTransfer = transferEvents[0];
            if (latestTransfer.functionName.includes('renounceOwnership')) {
              owner = 'RENOUNCED';
              console.log('Ownership renounced according to transaction events');
            } else {
              // Extract the new owner from the input data
              const inputData = latestTransfer.input;
              if (inputData && inputData.length >= 74) {
                // Extract the address from the input data (typical format for transferOwnership)
                const newOwner = '0x' + inputData.substring(34, 74);
                if (web3.utils.isAddress(newOwner)) {
                  owner = newOwner;
                  console.log(`Latest owner from transfer events: ${owner}`);
                }
              }
            }
          }
        }
      } catch (error) {
        console.error('Error checking ownership transfer events:', error.message);
      }
    }
    
    // Get token contract creation date
    const creationInfo = await getContractCreationInfo(address);
    
    // If owner is still Unknown, use the contract creator as owner
    if (owner === 'Unknown' && creationInfo && creationInfo.creator) {
      owner = creationInfo.creator;
      console.log(`Using contract creator as owner: ${owner}`);
    }
    
    // Get LP info for Ethereum
    const lpInfo = await getLPInfo(address);
    
    // Get token age
    const age = moment().diff(moment.unix(creationInfo.timestamp), 'seconds');
    const ageFormatted = formatAge(age);
    
    // Get tax information (buy/sell)
    const taxInfo = await getTaxInfo(address);
    
    // Removed holders data collection
    
    return {
      name,
      symbol,
      decimals,
      totalSupply,
      owner,
      contractAddress: address,
      creationDate: creationInfo.timestamp,
      ageFormatted,
      lpInfo,
      taxInfo
    };
  } catch (error) {
    console.error('Error getting ETH token details:', error);
    throw new Error('Failed to get ETH token details');
  }
}

/**
 * Get contract creation information
 */
async function getContractCreationInfo(address) {
  try {
    const response = await axios.get(ETHERSCAN_API, {
      params: {
        module: 'contract',
        action: 'getcontractcreation',
        contractaddresses: address,
        apikey: ETH_API_KEY
      }
    });
    
    if (response.data.status === '1') {
      // Get block timestamp for creation date
      const txHash = response.data.result[0].txHash;
      const txInfo = await web3.eth.getTransaction(txHash);
      const block = await web3.eth.getBlock(txInfo.blockNumber);
      
      return {
        creator: response.data.result[0].contractCreator,
        txHash,
        timestamp: Number(block.timestamp) // Convert BigInt to Number
      };
    }
    
    return { timestamp: Math.floor(Date.now() / 1000) - 86400 }; // Default to 1 day ago
  } catch (error) {
    console.error('Error getting contract creation info:', error);
    return { timestamp: Math.floor(Date.now() / 1000) - 86400 }; // Default to 1 day ago
  }
}

/**
 * Simplify the top holders function - not used anymore
 */
async function getTopHolders(address) {
  return { count: 0, top: [] };
}

/**
 * Get LP information - Typically Uniswap for Ethereum
 */
async function getLPInfo(address) {
  // This would normally involve checking Uniswap or other DEXes
  // Simplified implementation for now
  return {
    dex: 'Uniswap V3',
    lpAddress: 'Unknown', // Would need to fetch from Uniswap subgraph
    liquidity: 'Unknown',
    locked: 'Unknown'
  };
}

/**
 * Get tax information
 */
async function getTaxInfo(address) {
  // This would require analyzing the token contract for transfer fees
  // Simplified implementation for now
  return {
    buy: '0.0%',
    sell: '0.0%'
  };
}

/**
 * Format age to human-readable string
 */
function formatAge(seconds) {
  const days = Math.floor(seconds / 86400);
  const hours = Math.floor((seconds % 86400) / 3600);
  const minutes = Math.floor((seconds % 3600) / 60);
  
  return `${days}d - ${hours}h ${minutes}m ${seconds % 60}s`;
}

module.exports = {
  getEthTokenDetails
};
