require('dotenv').config();
const { Web3 } = require('web3');  // Updated import style
const axios = require('axios');
const moment = require('moment');

const POLYGON_RPC = process.env.POLYGON_RPC || 'https://polygon-rpc.com';
const POLYGON_API_KEY = process.env.POLYGON_API_KEY;
const POLYGONSCAN_API = 'https://api.polygonscan.com/api';
const web3 = new Web3(POLYGON_RPC);

// Standard ERC20 ABI (same as other EVM chains)
const ERC20_ABI = [
  { "inputs": [], "name": "name", "outputs": [{ "type": "string" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "symbol", "outputs": [{ "type": "string" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "decimals", "outputs": [{ "type": "uint8" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "totalSupply", "outputs": [{ "type": "uint256" }], "stateMutability": "view", "type": "function" },
  { "inputs": [{ "type": "address" }], "name": "balanceOf", "outputs": [{ "type": "uint256" }], "stateMutability": "view", "type": "function" },
  { "inputs": [], "name": "owner", "outputs": [{ "type": "address" }], "stateMutability": "view", "type": "function" }
];

async function getPolygonTokenDetails(address) {
  try {
    const contract = new web3.eth.Contract(ERC20_ABI, address);
    
    // Basic token info
    const [name, symbol, decimalsStr, totalSupplyWei] = await Promise.all([
      contract.methods.name().call(),
      contract.methods.symbol().call(),
      contract.methods.decimals().call(),
      contract.methods.totalSupply().call()
    ]);
    
    const decimals = parseInt(decimalsStr);
    const totalSupply = Number(totalSupplyWei) / Math.pow(10, decimals);
    
    // Get owner address (may fail if not implemented)
    let owner = 'Unknown';
    try {
      owner = await contract.methods.owner().call();
      
      // Check if owner is null address (renounced)
      if (owner === '0x0000000000000000000000000000000000000000') {
        owner = 'RENOUNCED';
      }
    } catch (error) {
      console.log('Owner method not available');
    }
    
    // Get token contract creation date
    const creationInfo = await getContractCreationInfo(address);
    
    // Get top holders
    const holders = await getTopHolders(address);
    
    // Check LP info - Typically QuickSwap for Polygon
    const lpInfo = await getLPInfo(address);
    
    // Get token age
    const age = moment().diff(moment.unix(creationInfo.timestamp), 'seconds');
    const ageFormatted = formatAge(age);
    
    // Get tax information (buy/sell)
    const taxInfo = await getTaxInfo(address);
    
    return {
      name,
      symbol,
      decimals,
      totalSupply,
      owner,
      contractAddress: address,
      creationDate: creationInfo.timestamp,
      ageFormatted,
      holders,
      lpInfo,
      taxInfo
    };
  } catch (error) {
    console.error('Error getting Polygon token details:', error);
    throw new Error('Failed to get Polygon token details');
  }
}

/**
 * Get contract creation information
 */
async function getContractCreationInfo(address) {
  try {
    const response = await axios.get(POLYGONSCAN_API, {
      params: {
        module: 'contract',
        action: 'getcontractcreation',
        contractaddresses: address,
        apikey: POLYGON_API_KEY
      }
    });
    
    if (response.data.status === '1') {
      // Get block timestamp for creation date
      const txHash = response.data.result[0].txHash;
      const txInfo = await web3.eth.getTransaction(txHash);
      const block = await web3.eth.getBlock(txInfo.blockNumber);
      
      return {
        creator: response.data.result[0].contractCreator,
        txHash,
        timestamp: block.timestamp
      };
    }
    
    return { timestamp: Math.floor(Date.now() / 1000) - 86400 }; // Default to 1 day ago
  } catch (error) {
    console.error('Error getting contract creation info:', error);
    return { timestamp: Math.floor(Date.now() / 1000) - 86400 }; // Default to 1 day ago
  }
}

/**
 * Get top token holders
 */
async function getTopHolders(address) {
  try {
    const response = await axios.get(POLYGONSCAN_API, {
      params: {
        module: 'token',
        action: 'tokenholderlist',
        contractaddress: address,
        apikey: POLYGON_API_KEY
      }
    });
    
    if (response.data.status === '1') {
      const holders = response.data.result.slice(0, 5).map(holder => ({
        address: holder.TokenHolderAddress,
        balance: holder.TokenHolderQuantity,
        percentage: parseFloat(holder.TokenHolderQuantity) / parseFloat(response.data.result[0].TokenHolderQuantity) * 100
      }));
      
      return {
        count: response.data.result.length,
        top: holders
      };
    }
    
    return { count: 0, top: [] };
  } catch (error) {
    console.error('Error getting token holders:', error);
    return { count: 0, top: [] };
  }
}

/**
 * Get LP information - Typically QuickSwap for Polygon
 */
async function getLPInfo(address) {
  // This would normally involve checking QuickSwap or other DEXes
  // Simplified implementation for now
  return {
    dex: 'QuickSwap',
    lpAddress: 'Unknown', // Would need to fetch from QuickSwap
    liquidity: 'Unknown',
    locked: 'Unknown'
  };
}

/**
 * Get tax information
 */
async function getTaxInfo(address) {
  // This would require analyzing the token contract for transfer fees
  // Simplified implementation for now
  return {
    buy: '0.0%',
    sell: '0.0%'
  };
}

/**
 * Format age to human-readable string
 */
function formatAge(seconds) {
  const days = Math.floor(seconds / 86400);
  const hours = Math.floor((seconds % 86400) / 3600);
  const minutes = Math.floor((seconds % 3600) / 60);
  
  return `${days}d - ${hours}h ${minutes}m ${seconds % 60}s`;
}

module.exports = {
  getPolygonTokenDetails
};
