/**
 * Security analyzer for checking token safety
 * This module evaluates various security aspects of a token
 */

const { checkHoneypotStatus } = require('../utils/honeypotDetector');

/**
 * Check if a token is likely to be secure based on various factors
 * @param {Object} tokenDetails - Token details from the chain-specific service
 * @param {Object} dexData - Token data from DEX Screener
 * @returns {Object} Security analysis results
 */
async function analyzeTokenSecurity(tokenDetails, dexData) {
  // Detect network to determine analysis method
  const network = getTokenNetwork(tokenDetails, dexData);
  
  // Score initialization
  const securityScores = {
    ownership: analyzeOwnership(tokenDetails),
    liquidity: analyzeLiquidity(tokenDetails, dexData),
    holderDistribution: analyzeHolderDistribution(tokenDetails),
    contractAge: analyzeContractAge(tokenDetails),
    taxes: analyzeTaxes(tokenDetails),
    overall: 0
  };

  // Special case for Solana tokens using SolSniffer data when available
  const isSolana = network === 'SOL' || 
                  (tokenDetails.contractAddress && 
                   !tokenDetails.contractAddress.startsWith('0x') && 
                   tokenDetails.hasOwnProperty('solSnifferScore'));
                   
  if (isSolana) {
    console.log('Using Solana-specific security analysis (skipping honeypot checks)');
    
    // Adjust scores based on SolSniffer data if available
    if (tokenDetails.solSnifferScore) {
      securityScores.overall = tokenDetails.solSnifferScore;
      
      // Determine security status based on SolSniffer score
      let securityStatus;
      if (tokenDetails.solSnifferScore >= 80) {
        securityStatus = 'SAFE';
      } else if (tokenDetails.solSnifferScore >= 65) {
        securityStatus = 'LIKELY SAFE';
      } else if (tokenDetails.solSnifferScore >= 50) {
        securityStatus = 'CAUTION';
      } else {
        securityStatus = 'HIGH RISK';
      }
      
      // Create warnings from security warnings in tokenDetails
      const securityWarnings = tokenDetails.securityWarnings || [];
      
      // Additional token-specific checks for Solana
      const contractFeatures = {
        isOpenSource: true, // Solana programs are open source
        isProxy: false,
        isMintable: tokenDetails.mintable === true,
        canPauseTrading: false,
        hasCooldown: false,
        cantSellAll: false,
        canModifyBalances: false,
        hasBlacklist: false,
        hasWhitelist: false,
        isAntiWhale: false,
        hasExternalCalls: false,
        hasHiddenOwner: false,
        canFreeze: tokenDetails.freezable === true
      };
      
      return {
        scores: securityScores,
        warnings: securityWarnings,
        status: securityStatus,
        emoji: getStatusEmoji(securityStatus),
        honeypot: { isHoneypot: false, confidence: 'N/A' }, // No honeypot checks for Solana
        contractFeatures: contractFeatures
      };
    } else {
      // Calculate overall score based on basic metrics for Solana tokens without SolSniffer
      securityScores.overall = calculateOverallScore(securityScores);
      const securityWarnings = generateSolanaWarnings(tokenDetails, dexData, securityScores);
      const securityStatus = determineSolanaSecurityStatus(securityScores.overall, securityWarnings);
      
      return {
        scores: securityScores,
        warnings: securityWarnings,
        status: securityStatus,
        emoji: getStatusEmoji(securityStatus),
        honeypot: { isHoneypot: false, confidence: 'N/A' }, // No honeypot for Solana
        contractFeatures: {
          isOpenSource: true,
          isMintable: false,
          canFreeze: false
        }
      };
    }
  }

  // For non-Solana tokens, continue with regular honeypot checks
  let honeypotStatus;
  try {
    // Only check honeypot for non-Solana tokens
    honeypotStatus = await checkHoneypotStatus(tokenDetails.contractAddress, network);
    console.log(`Honeypot check result: ${JSON.stringify(honeypotStatus)}`);
    
    // If tax information is available from honeypot check but not in tokenDetails
    if (honeypotStatus.buyTax !== null && honeypotStatus.sellTax !== null) {
      if (!tokenDetails.taxInfo) {
        tokenDetails.taxInfo = {};
      }
      if (!tokenDetails.taxInfo.buy || tokenDetails.taxInfo.buy === '?') {
        tokenDetails.taxInfo.buy = `${honeypotStatus.buyTax.toFixed(1)}%`;
      }
      if (!tokenDetails.taxInfo.sell || tokenDetails.taxInfo.sell === '?') {
        tokenDetails.taxInfo.sell = `${honeypotStatus.sellTax.toFixed(1)}%`;
      }
      
      // Recalculate tax score with the new information
      securityScores.taxes = analyzeTaxes(tokenDetails);
    }
  } catch (error) {
    console.error('Error during honeypot check:', error);
    honeypotStatus = {
      isHoneypot: false,
      confidence: 'Error',
      warning: `API error: ${error.message}`
    };
  }

  // Extract or derive contract features from tokenDetails and honeypot checks
  const contractFeatures = {
    isOpenSource: true, // Default most contracts to open source
    isProxy: false,
    isMintable: false,
    canPauseTrading: false,
    hasCooldown: false,
    cantSellAll: honeypotStatus?.cantSellAll || false,
    canModifyBalances: false,
    hasBlacklist: false,
    hasWhitelist: false,
    isAntiWhale: false,
    hasExternalCalls: false,
    hasHiddenOwner: false
  };
  
  // Update with honeypot API insights if available
  if (honeypotStatus) {
    // Update features based on honeypot detection
    if (honeypotStatus.warning) {
      if (honeypotStatus.warning.includes('blacklist')) contractFeatures.hasBlacklist = true;
      if (honeypotStatus.warning.includes('whitelist')) contractFeatures.hasWhitelist = true;
      if (honeypotStatus.warning.includes('cooldown')) contractFeatures.hasCooldown = true;
      if (honeypotStatus.warning.includes('cannot sell')) contractFeatures.cantSellAll = true;
      if (honeypotStatus.warning.includes('balance')) contractFeatures.canModifyBalances = true;
      if (honeypotStatus.warning.includes('mint')) contractFeatures.isMintable = true;
      if (honeypotStatus.warning.includes('pause')) contractFeatures.canPauseTrading = true;
      if (honeypotStatus.warning.includes('proxy')) contractFeatures.isProxy = true;
      if (honeypotStatus.warning.includes('anti-whale')) contractFeatures.isAntiWhale = true;
    }
  }

  // Calculate overall score (weighted average)
  securityScores.overall = calculateOverallScore(securityScores);

  // Adjust security score if honeypot
  if (honeypotStatus && honeypotStatus.isHoneypot) {
    securityScores.overall = Math.max(10, securityScores.overall - 50); // Heavily penalize honeypots
  }

  const securityWarnings = generateWarnings(tokenDetails, dexData, securityScores, honeypotStatus);
  const securityStatus = determineSecurityStatus(securityScores.overall, securityWarnings, honeypotStatus);

  return {
    scores: securityScores,
    warnings: securityWarnings,
    status: securityStatus,
    emoji: getStatusEmoji(securityStatus),
    honeypot: honeypotStatus, // Add honeypot status to the result
    contractFeatures: contractFeatures // Add contract features to the analysis results
  };
}

/**
 * Get the network identifier for the token
 */
function getTokenNetwork(tokenDetails, dexData) {
  // First check if the token is a Solana token (non-0x address)
  if (tokenDetails.contractAddress && !tokenDetails.contractAddress.startsWith('0x')) {
    return 'SOL';
  }

  // Try to get network from dexData first
  if (dexData && dexData.chainId) {
    const chainId = dexData.chainId.toLowerCase();
    if (chainId.includes('bsc')) return 'BSC';
    if (chainId.includes('eth')) return 'ETH';
    if (chainId.includes('polygon')) return 'POLYGON';
    if (chainId.includes('avax')) return 'AVALANCHE';
    if (chainId.includes('sol')) return 'SOL';
  }
  
  // Fallback to contract address format detection for EVM chains
  const address = tokenDetails.contractAddress.toLowerCase();
  if (address.startsWith('0x')) {
    // For EVM chains, check contractAddress first two characters
    return 'BSC'; // Default to BSC for EVM addresses
  }
  
  return 'Unknown';
}

/**
 * Analyze token ownership
 */
function analyzeOwnership(tokenDetails) {
  // Highest score if ownership is renounced
  if (tokenDetails.owner === 'RENOUNCED') {
    return 100;
  }
  
  // If owner is identified but not renounced, medium score
  if (tokenDetails.owner !== 'Unknown' && tokenDetails.owner !== 'RENOUNCED') {
    return 50;
  }
  
  // Unknown ownership is a potential risk
  return 30;
}

/**
 * Analyze token liquidity
 */
function analyzeLiquidity(tokenDetails, dexData) {
  if (!dexData || !dexData.liquidityUSD) {
    return 0;
  }
  
  const liquidityUSD = parseFloat(dexData.liquidityUSD);
  
  // Higher liquidity means higher score
  if (liquidityUSD > 1000000) return 100; // >$1M
  if (liquidityUSD > 500000) return 90;   // >$500k
  if (liquidityUSD > 100000) return 70;   // >$100k
  if (liquidityUSD > 50000) return 50;    // >$50k
  if (liquidityUSD > 10000) return 30;    // >$10k
  
  return 10; // Low liquidity
}

/**
 * Analyze holder distribution - modified to return default score
 */
function analyzeHolderDistribution(tokenDetails) {
  // Since holder data has been removed, always return a default score
  return 50; // Default/middle score
}

/**
 * Analyze contract age
 */
function analyzeContractAge(tokenDetails) {
  if (!tokenDetails.creationDate) {
    return 50; // Default score if no data
  }
  
  const ageInDays = (Date.now()/1000 - tokenDetails.creationDate) / 86400;
  
  // Older contracts are generally more trustworthy
  if (ageInDays > 365) return 100; // >1 year
  if (ageInDays > 180) return 90;  // >6 months
  if (ageInDays > 90) return 80;   // >3 months
  if (ageInDays > 30) return 70;   // >1 month
  if (ageInDays > 7) return 50;    // >1 week
  if (ageInDays > 3) return 30;    // >3 days
  if (ageInDays > 1) return 20;    // >1 day
  
  return 10; // <1 day old (very new)
}

/**
 * Analyze tax structure
 */
function analyzeTaxes(tokenDetails) {
  if (!tokenDetails.taxInfo) {
    return 50; // Default score if no data
  }
  
  const buyTax = parseFloat(tokenDetails.taxInfo.buy) || 0;
  const sellTax = parseFloat(tokenDetails.taxInfo.sell) || 0;
  
  const totalTax = buyTax + sellTax;
  
  // Lower taxes are better
  if (totalTax === 0) return 100;   // No tax
  if (totalTax <= 5) return 90;     // Low tax
  if (totalTax <= 10) return 70;    // Moderate tax
  if (totalTax <= 20) return 50;    // High tax
  if (totalTax <= 30) return 30;    // Very high tax
  
  return 10; // Extreme tax
}

/**
 * Calculate overall security score
 */
function calculateOverallScore(scores) {
  // Weighted calculation
  const weights = {
    ownership: 0.3,
    liquidity: 0.2,
    holderDistribution: 0.2,
    contractAge: 0.15,
    taxes: 0.15
  };
  
  let overallScore = 0;
  let totalWeight = 0;
  
  for (const [key, score] of Object.entries(scores)) {
    if (key !== 'overall' && weights[key]) {
      overallScore += score * weights[key];
      totalWeight += weights[key];
    }
  }
  
  return Math.round(overallScore / totalWeight);
}

/**
 * Generate warnings based on security analysis
 */
function generateWarnings(tokenDetails, dexData, scores, honeypotStatus) {
  const warnings = [];
  
  // Add honeypot warning first if relevant
  if (honeypotStatus && honeypotStatus.isHoneypot) {
    let honeypotWarning = '🚨 POTENTIAL HONEYPOT';
    if (honeypotStatus.warning) {
      honeypotWarning += `: ${honeypotStatus.warning}`;
    }
    if (honeypotStatus.source) {
      honeypotWarning += ` [Source: ${honeypotStatus.source}]`;
    }
    warnings.push(honeypotWarning);
  }
  
  // Ownership warnings
  if (tokenDetails.owner !== 'RENOUNCED') {
    warnings.push('🔶 Contract ownership not renounced');
  }
  
  // Liquidity warnings
  if (!dexData || !dexData.liquidityUSD || parseFloat(dexData.liquidityUSD) < 50000) {
    // Check if low liquidity is the only concern (which is not a honeypot indicator)
    if (!honeypotStatus || !honeypotStatus.isHoneypot) {
      warnings.push('🔶 Low liquidity may cause high slippage');
    }
  }
  
  // Holder distribution warnings removed
  
  // Age warnings
  if (scores.contractAge < 50) {
    warnings.push('🔶 Contract is relatively new');
  }
  
  // Tax warnings
  if (tokenDetails.taxInfo) {
    const buyTax = parseFloat(tokenDetails.taxInfo.buy) || 0;
    const sellTax = parseFloat(tokenDetails.taxInfo.sell) || 0;
    
    if (buyTax + sellTax > 10) {
      warnings.push(`🔶 High combined tax: ${buyTax + sellTax}%`);
    }
  }
  
  return warnings;
}

/**
 * Generate warnings specific to Solana tokens
 */
function generateSolanaWarnings(tokenDetails, dexData, scores) {
  const warnings = [];
  
  // Liquidity warnings
  if (!dexData || !dexData.liquidityUSD || parseFloat(dexData.liquidityUSD) < 25000) {
    warnings.push('🔶 Low liquidity may cause high slippage');
  }
  
  // Age warnings
  if (scores.contractAge < 50) {
    warnings.push('🔶 Token is relatively new');
  }
  
  // Add minting/freezing warnings if available from token details
  if (tokenDetails.mintable === true) {
    warnings.push('🔶 Token supply can be increased (mintable)');
  }
  
  if (tokenDetails.freezable === true) {
    warnings.push('🔶 Token accounts can be frozen by authority');
  }
  
  return warnings;
}

/**
 * Determine overall security status considering honeypot status
 */
function determineSecurityStatus(overallScore, warnings, honeypotStatus) {
  // If token is a confirmed honeypot, always return HIGH RISK regardless of other factors
  if (honeypotStatus && honeypotStatus.isHoneypot && 
      (honeypotStatus.confidence === 'High' || honeypotStatus.confidence === 'Medium')) {
    return 'HIGH RISK';
  }
  
  // Normal security status determination
  if (overallScore >= 80 && warnings.length <= 1) return 'SAFE';
  if (overallScore >= 60) return 'LIKELY SAFE';
  if (overallScore >= 40) return 'CAUTION';
  return 'HIGH RISK';
}

/**
 * Determine security status for Solana tokens
 */
function determineSolanaSecurityStatus(overallScore, warnings) {
  // More lenient thresholds for Solana since we don't have honeypot checks
  if (overallScore >= 75 && warnings.length <= 1) return 'SAFE';
  if (overallScore >= 55) return 'LIKELY SAFE';
  if (overallScore >= 35) return 'CAUTION';
  return 'HIGH RISK';
}

/**
 * Get emoji for security status
 */
function getStatusEmoji(status) {
  switch (status) {
    case 'SAFE': return '🟢';
    case 'LIKELY SAFE': return '🟢';
    case 'CAUTION': return '🟡';
    case 'HIGH RISK': return '🔴';
    default: return '⚪';
  }
}

module.exports = {
  analyzeTokenSecurity
};
