const axios = require('axios');
const cheerio = require('cheerio');

/**
 * Check if a DEX is likely to be paid based on various indicators
 * Enhanced with additional API checks and pattern recognition
 */
function checkDexPaidStatus(mainPair) {
  try {
    // 1. Check API-provided flags directly
    if (mainPair.flags) {
      // Look for paid indicators in flags array
      if (Array.isArray(mainPair.flags)) {
        const paidFlags = mainPair.flags.filter(flag => 
          flag.toLowerCase().includes('promoted') || 
          flag.toLowerCase().includes('sponsored') ||
          flag.toLowerCase().includes('verified') ||
          flag.toLowerCase().includes('paid')
        );
        
        if (paidFlags.length > 0) {
          console.log(`DEX Paid detected via flags: ${paidFlags.join(', ')}`);
          return true;
        }
      }
      // Look for paid indicators in flags object
      else if (typeof mainPair.flags === 'object') {
        for (const key in mainPair.flags) {
          const value = mainPair.flags[key];
          if ((typeof value === 'string' && value.toLowerCase().includes('promoted')) ||
              key.toLowerCase().includes('promoted') ||
              key.toLowerCase().includes('paid') ||
              key.toLowerCase().includes('verified')) {
            console.log(`DEX Paid detected via flags object: ${key}`);
            return true;
          }
        }
      }
    }
    
    // 2. Check for promoted/paid indicators in labels
    if (mainPair.labels && Array.isArray(mainPair.labels)) {
      const paidLabels = mainPair.labels.filter(label => 
        label.toLowerCase().includes('promoted') || 
        label.toLowerCase().includes('sponsored') || 
        label.toLowerCase().includes('paid') ||
        label.toLowerCase().includes('verified')
      );
      
      if (paidLabels.length > 0) {
        console.log(`DEX Paid detected via labels: ${paidLabels.join(', ')}`);
        return true;
      }
    }
    
    // 3. Check major DEXes with typical listing fees
    const knownPaidDexes = [
      'pancakeswap', 'uniswap', 'sushiswap', 'traderjoe', 'raydium', 
      'quickswap', 'spookyswap', 'spiritswap', 'balancer'
    ];
    
    if (mainPair.dexId && knownPaidDexes.some(dex => mainPair.dexId.toLowerCase().includes(dex))) {
      console.log(`DEX Paid likelihood based on major DEX: ${mainPair.dexId}`);
      
      // Additional verification for major DEXes: Check if liquidity is high
      const liquidity = parseFloat(mainPair.liquidity?.usd || 0);
      if (liquidity > 100000) {  // Over $100k suggests legitimacy/payment
        console.log(`High liquidity ($${liquidity}) indicates likely paid listing`);
        return true;
      }
    }
    
    // 4. Check if token info is complete and verified
    if (mainPair.baseToken) {
      // Complete token details usually indicate verified/paid status
      if (mainPair.baseToken.name && 
          mainPair.baseToken.symbol && 
          mainPair.baseToken.address &&
          mainPair.audit === true) {
        console.log(`Complete token data with audit flag suggests paid/verified status`);
        return true;
      }
    }
    
    // 5. Check additional API data that might indicate paid status
    if (mainPair.audit === true || 
        (mainPair.trust && mainPair.trust > 0.7) ||
        mainPair.verified === true) {
      console.log(`Trust indicators suggest paid/verified status`);
      return true;
    }
    
    // For logging purposes, record raw data we're analyzing
    console.log(`DEX Paid detection analyzed: dexId=${mainPair.dexId}, ` +
                `liquidity=${mainPair.liquidity?.usd}, ` +
                `labels=${mainPair.labels ? JSON.stringify(mainPair.labels) : 'none'}`);
    
    return false;
  } catch (error) {
    console.error('Error in DEX paid detection:', error.message);
    return false;  // Default to false on error
  }
}

// Add function to check paid status with external API
async function checkExternalPaidApi(tokenAddress, network) {
  try {
    // This can be expanded to check other external APIs
    // Currently focusing on DexScreener data
    
    const apiUrl = `https://api.dexscreener.com/latest/dex/tokens/${tokenAddress}`;
    const response = await axios.get(apiUrl, {
      headers: {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) Chrome/120.0.0.0'
      },
      timeout: 10000
    });
    
    if (!response.data || !response.data.pairs || response.data.pairs.length === 0) {
      console.log('No pairs found in external API for paid detection');
      return false;
    }
    
    // Check multiple pairs for the token
    for (const pair of response.data.pairs) {
      if (checkDexPaidStatus(pair)) {
        console.log(`External API detected paid status for pair ${pair.pairAddress}`);
        return true;
      }
    }
    
    return false;
  } catch (error) {
    console.error('Error checking external paid API:', error.message);
    return false;
  }
}

/**
 * Fetch token data from DexScreener with retry capability
 */
async function fetchDexScreenerData(address, network) {
  try {
    // Use multiple timeouts and retries for resilience
    const MAX_RETRIES = 3;
    let retryCount = 0;
    let lastError;

    while (retryCount < MAX_RETRIES) {
      try {
  // Use the DexScreener API URL
  const apiUrl = `https://api.dexscreener.com/latest/dex/tokens/${address}`;
  console.log(`Fetching DexScreener data from: ${apiUrl} (attempt ${retryCount + 1}/${MAX_RETRIES})`);
        
        const response = await axios.get(apiUrl, {
          headers: {
            'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) Chrome/120.0.0.0'
          },
          timeout: 15000 // 15 second timeout
        });
        
        if (!response.data || !response.data.pairs || response.data.pairs.length === 0) {
          console.log('No pair data found');
          return null;
        }
        
        // Filter pair berdasarkan network
        const normalizedNetwork = 
          network.toLowerCase() === 'eth' ? 'ethereum' : 
          network.toLowerCase() === 'sol' ? 'solana' : 
          network.toLowerCase();
        
  console.log(`Searching for pairs for network: ${normalizedNetwork}`);
        
        let pairs = response.data.pairs.filter(pair => 
          pair.chainId && pair.chainId.toLowerCase() === normalizedNetwork.toLowerCase()
        );
        
        // Use all pairs if none match the requested network
        if (pairs.length === 0) {
          console.log(`No pairs found for network ${normalizedNetwork}, using all pairs`);
          pairs = response.data.pairs;
        }
        
  console.log(`Found ${pairs.length} pairs for this token`);
        
        // Special handling for Solana network
        if (network.toLowerCase() === 'sol' || normalizedNetwork === 'solana') {
          console.log('Using special handling for Solana tokens');
          
          // Sort based on liquidity, volume, or recency for Solana tokens
          pairs.sort((a, b) => {
            // First prioritize the proper chain
            if (a.chainId.toLowerCase() === 'solana' && b.chainId.toLowerCase() !== 'solana') return -1;
            if (a.chainId.toLowerCase() !== 'solana' && b.chainId.toLowerCase() === 'solana') return 1;
            
            // Then prioritize by liquidity
            return (parseFloat(b.liquidity?.usd || 0) - parseFloat(a.liquidity?.usd || 0));
          });
        } else {
          // Standard sorting for other chains
          pairs.sort((a, b) => (parseFloat(b.liquidity?.usd) || 0) - (parseFloat(a.liquidity?.usd) || 0));
        }
        
        const mainPair = pairs[0];
        
        // Log raw data for debugging and verification
        console.log(`Raw data from DexScreener:`, {
          mcap: mainPair.mcap,
          fdv: mainPair.fdv,
          liquidity: mainPair.liquidity?.usd,
          price: mainPair.priceUsd
        });
        
  // IMPORTANT: Use values directly from DexScreener without conversion
  // to ensure the same figures as shown on DexScreener
        
        // Extract social links directly from API response if available - DEX Paid tokens often have this
        const socialLinks = {
          telegram: null,
          twitter: null,
          website: null
        };
        
        // Check if the token has DEX paid status with enhanced info
        if (mainPair.info) {
          console.log('Found info object in API response - possibly DEX Paid token');
          
          // Extract websites
          if (mainPair.info.websites && Array.isArray(mainPair.info.websites)) {
            for (const website of mainPair.info.websites) {
              // Check if website is a string before using it
              if (typeof website === 'string') {
                console.log(`Website found in API info: ${website}`);
                socialLinks.website = website;
                break; // Just use the first one
              } else if (website && typeof website === 'object' && website.url) {
                // Handle case where website is an object with a url property
                console.log(`Website found in API info (object format): ${website.url}`);
                socialLinks.website = website.url;
                break;
              } else {
                console.log(`Website found in API info (unknown format):`, website);
              }
            }
          }
          
          // Extract socials from API response with proper type checking
          if (mainPair.info.socials && Array.isArray(mainPair.info.socials)) {
            for (const social of mainPair.info.socials) {
              if (!social) continue;
              
              // Handle different social media format structures
              let url = '';
              
              if (typeof social === 'string') {
                url = social.toLowerCase();
                console.log(`Social link found in API info: ${url}`);
              } else if (typeof social === 'object') {
                // It might be an object with a url property
                if (social.url && typeof social.url === 'string') {
                  url = social.url.toLowerCase();
                  console.log(`Social link found in API info (object format): ${url}`);
                } else {
                  console.log(`Social object found but in unknown format:`, social);
                  continue; // Skip this iteration
                }
              } else {
                console.log(`Unsupported social type: ${typeof social}`);
                continue; // Skip this iteration
              }
              
              // Now check the URL to categorize the social media link
              if (url.includes('t.me/') || url.includes('telegram')) {
                socialLinks.telegram = typeof social === 'string' ? social : (social.url || null);
              } else if (url.includes('twitter.com/') || url.includes('x.com/')) {
                socialLinks.twitter = typeof social === 'string' ? social : (social.url || null);
              }
            }
          }
        }
        
  // Scrape the DexScreener page for additional info
  console.log(`Scraping DexScreener page for additional info...`);
        const pairUrl = `https://dexscreener.com/${mainPair.chainId}/${mainPair.pairAddress}`;
        const webData = await scrapeDexScreenerPage(pairUrl);
        
        // Cek status LP Lock
        let lpLockStatus = 'Unknown';
        
        // First, check for direct lock status in the liquidity field
        if (mainPair.liquidity) {
          if (mainPair.liquidity.locked === true) {
            console.log('LP is locked according to DexScreener API (locked: true)');
            lpLockStatus = 'LOCKED';
          }
          else if (typeof mainPair.liquidity.lockStatus === 'string' && 
                  mainPair.liquidity.lockStatus.toLowerCase() === 'locked') {
            console.log('LP is locked according to DexScreener API (lockStatus: locked)');
            lpLockStatus = 'LOCKED';
          }
          else if (mainPair.liquidity.lockedUsd && 
                  parseFloat(mainPair.liquidity.lockedUsd) > 0) {
            console.log(`LP has lockedUsd value: ${mainPair.liquidity.lockedUsd}`);
            lpLockStatus = 'LOCKED';
          }
          
          // Log all liquidity data for debugging
          console.log('DexScreener Liquidity Data:', JSON.stringify(mainPair.liquidity, null, 2));
        }
        
        const ownerInfo = webData.owner || null;
        
        // First check with the external API
        const isPaidExternal = await checkExternalPaidApi(address, network);
        
        // Check with our own detection
        const isPaidInternal = checkDexPaidStatus(mainPair);
        
        // Combine results - if either method detects paid status
        const isPaid = isPaidExternal || isPaidInternal;
        console.log(`DEX Paid Status: ${isPaid ? 'PAID' : 'Not detected'} (External: ${isPaidExternal}, Internal: ${isPaidInternal})`);
        
        // Check if the contract is verified
        let isVerified = false;
        
        // Check for verification indicators in the API data
        if (mainPair.info && mainPair.info.verified === true) {
          isVerified = true;
        } else if (mainPair.verified === true) {
          isVerified = true;
        } else if (mainPair.audit === true) {
          isVerified = true;
        } else if (webData && webData.isVerified === true) {
          isVerified = true;
        }
        
        console.log(`Contract Verification Status: ${isVerified ? 'Verified' : 'Not verified'}`);
        
        // Merge social links, prioritizing API data over scraped data
        const mergedSocials = {
          telegram: socialLinks.telegram || webData.socials.telegram,
          twitter: socialLinks.twitter || webData.socials.twitter,
          website: socialLinks.website || webData.socials.website
        };
        
        console.log('Social links found:', mergedSocials);
        
        // Extract extended data from DexScreener API response for detailed info display
        const extendedDexData = {
          // Basic token info
          name: mainPair.baseToken.name,
          symbol: mainPair.baseToken.symbol,
          
          // Price data
          price: mainPair.priceUsd,
          priceNative: mainPair.priceNative,
          priceChange: {
            h1: mainPair.priceChange?.h1,
            h24: mainPair.priceChange?.h24,
            h6: mainPair.priceChange?.h6,
            m5: mainPair.priceChange?.m5,
          },
          
          // Volume data
          volume: {
            h24: mainPair.volume?.h24,
            h6: mainPair.volume?.h6,
            h1: mainPair.volume?.h1,
            m5: mainPair.volume?.m5,
          },
          
          // Transaction data
          txns: mainPair.txns ? {
            h24: {
              buys: mainPair.txns.h24?.buys || 0,
              sells: mainPair.txns.h24?.sells || 0,
              total: (mainPair.txns.h24?.buys || 0) + (mainPair.txns.h24?.sells || 0)
            },
            h6: {
              buys: mainPair.txns.h6?.buys || 0,
              sells: mainPair.txns.h6?.sells || 0,
              total: (mainPair.txns.h6?.buys || 0) + (mainPair.txns.h6?.sells || 0)
            },
            h1: {
              buys: mainPair.txns.h1?.buys || 0,
              sells: mainPair.txns.h1?.sells || 0,
              total: (mainPair.txns.h1?.buys || 0) + (mainPair.txns.h1?.sells || 0)
            }
          } : null,
          
          // Pair information
          pairAddress: mainPair.pairAddress,
          pairCreatedAt: mainPair.pairCreatedAt ? new Date(mainPair.pairCreatedAt) : null,
          baseToken: {
            address: mainPair.baseToken.address,
            name: mainPair.baseToken.name,
            symbol: mainPair.baseToken.symbol
          },
          quoteToken: {
            address: mainPair.quoteToken.address,
            name: mainPair.quoteToken.name,
            symbol: mainPair.quoteToken.symbol
          },
          
          // Dex information
          dexId: mainPair.dexId,
          chainId: mainPair.chainId,
          url: mainPair.url || `https://dexscreener.com/${mainPair.chainId}/${mainPair.pairAddress}`
        };
        
        // Extract holder information
        let holderCount = null;
        let lpHolderCount = null;
        
        try {
          // Try to get holder count from DexScreener data
          if (mainPair.txns && mainPair.txns.h24 && mainPair.txns.h24.holders) {
            holderCount = formatHolderCount(mainPair.txns.h24.holders);
          } else if (mainPair.holders) {
            holderCount = formatHolderCount(mainPair.holders);
          }
          
          // Try to find LP holder count if available
          if (mainPair.liquidityHolders) {
            lpHolderCount = mainPair.liquidityHolders;
          } else {
            // Default value for LP holders if not available
            lpHolderCount = 10; // Common estimate
          }
        } catch (error) {
          console.log(`Error extracting holder info: ${error.message}`);
        }
        
        // For Solana tokens, try to extract more specific data
        if (network.toLowerCase() === 'sol' || mainPair.chainId.toLowerCase() === 'solana') {
          console.log('Extracting enhanced data for Solana token');
          
          // For Solana tokens, gather more DEX information with proper names
          let dexName = mainPair.dexId || 'Unknown';
          
          // Normalize common Solana DEX names
          if (dexName.toLowerCase().includes('raydium')) dexName = 'Raydium';
          else if (dexName.toLowerCase().includes('orca')) dexName = 'Orca';
          else if (dexName.toLowerCase().includes('jupiter')) dexName = 'Jupiter';
          else if (dexName.toLowerCase().includes('serum')) dexName = 'Serum';
          
          // For Solana, supplement with market cap calculation if missing
          if ((!mainPair.mcap || isNaN(mainPair.mcap)) && mainPair.priceUsd) {
            try {
              // Attempt to get total supply from token info if available
              // This is a rough estimate
              const estimatedSupply = 1000000000; // Fallback estimate
              const estimatedMcap = parseFloat(mainPair.priceUsd) * estimatedSupply;
              mainPair.mcap = estimatedMcap;
              console.log(`Estimated market cap for Solana token: ${mainPair.mcap}`);
            } catch (err) {
              console.log('Could not estimate market cap:', err.message);
            }
          }
        }
        
        const { parseMarketCap } = require('./numberUtils');
        const marketCapValue = parseMarketCap(mainPair.mcap);

        return {
          name: mainPair.baseToken.name,
          symbol: mainPair.baseToken.symbol,
          price: mainPair.priceUsd,
          priceChange24h: mainPair.priceChange?.h24,
          // Gunakan nilai asli tanpa konversi
          liquidityUSD: mainPair.liquidity?.usd,
          volume24h: mainPair.volume?.h24,
          // Gunakan nilai FDV dan MCAP persis dari API
          fdv: mainPair.fdv,
          // Provide marketCap as a numeric value when possible (parsed from strings like "23.4M")
          marketCap: marketCapValue !== null ? marketCapValue : mainPair.mcap,
          pairAddress: mainPair.pairAddress,
          dexId: mainPair.dexId,
          chainId: mainPair.chainId, // Add chainId to help track which network the data came from
          lpLockStatus: lpLockStatus,
          // Add lock details for deeper information
          lpLockDetails: mainPair.liquidity ? {
            locked: mainPair.liquidity.locked,
            lockStatus: mainPair.liquidity.lockStatus,
            lockedUsd: mainPair.liquidity.lockedUsd,
            lockUrl: mainPair.liquidity.lockUrl || null,
            locker: mainPair.liquidity.locker || null,
            unlockDate: mainPair.liquidity.unlockDate || null
          } : null,
          ownerInfo: ownerInfo,
          socials: mergedSocials, // Add socials info
          url: pairUrl,
          isPaid: isPaid, // Add paid status indicator
          isVerified: isVerified, // Add verification status
          // Simpan raw data juga untuk debugging
          rawData: mainPair,
          // Add extended data
          extendedDexData: extendedDexData,
          holderCount: holderCount,
          lpHolderCount: lpHolderCount
        };
      } catch (error) {
        lastError = error;
        retryCount++;
        console.log(`DexScreener attempt ${retryCount} failed: ${error.message}`);
        
        // Wait before retry
        if (retryCount < MAX_RETRIES) {
          await new Promise(resolve => setTimeout(resolve, 2000)); // 2 second delay
        }
      }
    }
    
    // If we get here, all retries failed
    throw lastError;
  } catch (error) {
    console.error('Error fetching DexScreener data:', error.message);
    return null;
  }
}

/**
 * Format holder count with commas for thousands
 */
function formatHolderCount(count) {
  if (count === undefined || count === null) return null;
  
  // If already formatted with commas, return as is
  if (typeof count === 'string' && count.includes(',')) return count;
  
  // Format number with commas
  return count.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}

/**
 * Scrape DexScreener page untuk informasi tambahan
 */
async function scrapeDexScreenerPage(url) {
  try {
  console.log(`Scraping page: ${url}`);
    
    const response = await axios.get(url, {
      headers: {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) Chrome/120.0.0.0'
      },
      timeout: 15000
    });
    
    const $ = cheerio.load(response.data);
    
    // Hasil yang akan dikembalikan
    const result = {
      lpLock: null,
      owner: null,
      socials: {
        telegram: null,
        twitter: null,
        website: null
      }
    };
    
    // 1. Cari informasi LP Lock
  console.log('Searching for LP Lock information...');
    
    // Cari semua teks yang mungkin mengindikasikan LP Lock
    $('div, span, p, a, button').each((i, el) => {
      const text = $(el).text().trim().toLowerCase();
      
      // Cari teks yang menunjukkan LP Lock
      if (text.includes('liquidity locked') || 
          text.includes('lp locked') || 
          text.includes('lock lp') || 
          text.match(/liq.*lock/i) ||
          text.match(/lock.*liquidity/i)) {
        
  console.log(`✓ LP Lock text found: "${$(el).text().trim()}"`);
        result.lpLock = 'LOCKED';
        
        // Cari detailnya (sampai kapan, dll)
        const lockDetails = $(el).parent().text().trim();
        if (lockDetails.includes('until') || lockDetails.includes('for')) {
          result.lpLock = `LOCKED (${lockDetails})`;
        }
      }
    });
    
    // Cek apakah ada label "LP Locked"
    $('.text-green').each((i, el) => {
      const text = $(el).text().trim();
      if (text.toLowerCase().includes('lock')) {
  console.log(`✓ LP Lock label found: "${text}"`);
        result.lpLock = 'LOCKED';
      }
    });
    
    // Improve LP Lock detection by looking for more specific indicators
  console.log('Searching for LP Lock information with enhanced methods...');
    
    // 1. Check for liquidity lock explicitly
    const liquidityElements = [
      '*liquidityLocked*', '*liquidity locked*', '*LP locked*', '*lock LP*',
      '*lockedLiquidity*', '*locked liquidity*', '*LP lock*'
    ];
    
    for (const selector of liquidityElements) {
      const elements = $(`*:contains("${selector.replace(/\*/g, '')}")`);
      if (elements.length > 0) {
        console.log(`Found liquidity lock indicator: "${selector}"`);
        result.lpLock = 'LOCKED';
        break;
      }
    }
    
    // 2. Check specifically for lock information
    // Some pages have specific elements like "Liquidity is locked" text
    $('div, span, p').each((i, el) => {
      const text = $(el).text().toLowerCase();
      if (text.includes('liquidity') && text.includes('lock')) {
        console.log(`Found LP lock reference: "${$(el).text()}"`);
        result.lpLock = 'LOCKED';
      }
    });
    
    // 2. Cari informasi Owner
  console.log('Searching for Owner information...');
    
    // Cari teks yang menunjukkan renounced ownership
    $('div, span, p, a, button').each((i, el) => {
      const text = $(el).text().trim().toLowerCase();
      
      if (text.includes('renounced') || 
          text.includes('ownership renounced') || 
          text.includes('verified contract') ||
          text.includes('ownership transferred')) {
        
  console.log(`✓ Owner text found: "${$(el).text().trim()}"`);
        result.owner = 'RENOUNCED';
      }
      
      // Cari alamat owner
      if (text.includes('owner') || text.includes('creator')) {
        const addressMatch = $(el).parent().text().match(/0x[a-fA-F0-9]{40}/);
        if (addressMatch) {
          console.log(`✓ Owner address found: ${addressMatch[0]}`);
          result.owner = addressMatch[0];
        }
      }
    });
    
    // Cari social media links
  console.log('Searching for social media information...');
    
  // Search all links on the page
    $('a').each((i, el) => {
      const href = $(el).attr('href') || '';
      
      // Telegram
      if (href.includes('t.me/') || href.includes('telegram.me/')) {
  console.log(`✓ Telegram link found: ${href}`);
        result.socials.telegram = href;
      }
      
      // Twitter
      if (href.includes('twitter.com/') || href.includes('x.com/')) {
  console.log(`✓ Twitter link found: ${href}`);
        result.socials.twitter = href;
      }
      
      // Website (biasanya link dengan domain.com atau .io dll)
      if ((href.startsWith('http') || href.startsWith('https')) && 
          !href.includes('t.me/') && 
          !href.includes('telegram.me/') && 
          !href.includes('twitter.com/') && 
          !href.includes('x.com/') &&
          !href.includes('dexscreener.com')) {
  console.log(`✓ Website link found: ${href}`);
        result.socials.website = href;
      }
    });
    
    // Enhanced social media link extraction, especially for DEX paid tokens
  console.log('Searching for social media information with enhanced methods...');
    
    // Check for the enhanced profile section that DEX paid tokens have
    $('a[target="_blank"]').each((i, el) => {
      const href = $(el).attr('href') || '';
      const parent = $(el).parent().parent().text().toLowerCase(); // Check parent context
      
      // Enhanced detection for DEX paid profiles
      const isPaidProfileSection = parent.includes('profile') || 
                                  parent.includes('links') || 
                                  parent.includes('social');
      
      if (isPaidProfileSection) {
        console.log(`Inspecting link in paid profile section: ${href}`);
      }
      
      // Telegram
      if (href.includes('t.me/') || href.includes('telegram.me/')) {
  console.log(`✓ Telegram link found: ${href}`);
        result.socials.telegram = href;
      }
      
      // Twitter
      if (href.includes('twitter.com/') || href.includes('x.com/')) {
  console.log(`✓ Twitter link found: ${href}`);
        result.socials.twitter = href;
      }
      
      // Website (biasanya link dengan domain.com atau .io dll)
      if ((href.startsWith('http') || href.startsWith('https')) && 
          !href.includes('t.me/') && 
          !href.includes('telegram.me/') && 
          !href.includes('twitter.com/') && 
          !href.includes('x.com/') &&
          !href.includes('dexscreener.com')) {
  console.log(`✓ Website link found: ${href}`);
        result.socials.website = href;
      }
    });
    
    return result;
  } catch (error) {
    console.error(`Error scraping DexScreener page: ${error.message}`);
    return { lpLock: null, owner: null, socials: {} };
  }
}

module.exports = {
  fetchDexScreenerData,
  checkDexPaidStatus,
  checkExternalPaidApi  // Export the new function
};
