const WebSocket = require('ws');
const axios = require('axios');

/**
 * Fetch holder data directly from DexScreener - DIRECT API method
 */
async function getHoldersViaWebSocket(address, network) {
  try {
    console.log(`🔍 Searching for holder data for: ${address}`);
    
  // STEP 1: Fetch data from DexScreener API
  const apiUrl = `https://api.dexscreener.com/latest/dex/tokens/${address}`;
  console.log(`Accessing API: ${apiUrl}`);
    const response = await axios.get(apiUrl, {
      headers: {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) Chrome/120.0.0.0'
      },
      timeout: 10000
    });
    
    if (!response.data || !response.data.pairs || response.data.pairs.length === 0) {
      console.log('No pair data found');
      return { count: 0 };
    }
    
  // STEP 2: Filter pairs by network
    const normalizedNetwork = network.toLowerCase() === 'eth' ? 'ethereum' : network.toLowerCase();
    let relevantPairs = response.data.pairs.filter(p => p.chainId.toLowerCase() === normalizedNetwork);
    
    // If no pairs match the network, fall back to all pairs
    if (relevantPairs.length === 0) {
      console.log(`No pairs found for network ${normalizedNetwork}, using all pairs`);
      relevantPairs = response.data.pairs;
    }
    
  // STEP 3: Sort by highest liquidity
    relevantPairs.sort((a, b) => (parseFloat(b.liquidity?.usd) || 0) - (parseFloat(a.liquidity?.usd) || 0));
    const mainPair = relevantPairs[0];
    
  console.log(`Top pair: ${mainPair.dexId} on ${mainPair.chainId}`);
    
  // STEP 4: ATTEMPT TO EXTRACT HOLDER DATA FROM API RESPONSE
  let holderCount = 0;
    
    // Coba ambil dari txns.h24.holders (paling sering ada)
    if (mainPair.txns && mainPair.txns.h24 && mainPair.txns.h24.holders) {
      holderCount = mainPair.txns.h24.holders;
      console.log(`✓ Holder data found in txns.h24.holders: ${holderCount}`);
    }
    // Alternative: direct holders field
    else if (mainPair.holders && typeof mainPair.holders === 'number') {
      holderCount = mainPair.holders;
      console.log(`✓ Holder data found in holders field: ${holderCount}`);
    }
    
    return {
      count: holderCount,
      source: "Data API", // Hide DexScreener reference
      url: null // No URL shown
    };
  } catch (error) {
    console.error('Error fetching holder data:', error.message);
    return { count: 0 };
  }
}

module.exports = {
  getHoldersViaWebSocket
};
