/**
 * Utility for handling network errors and retrying requests
 */

/**
 * Retry a function up to specified number of times
 * @param {Function} fn - Async function to retry
 * @param {Number} retries - Number of retries
 * @param {Number} delay - Delay between retries in ms
 */
async function retryOperation(fn, retries = 3, delay = 1000) {
  let lastError;
  
  for (let i = 0; i < retries; i++) {
    try {
      return await fn();
    } catch (error) {
      console.log(`Operation failed, retrying (${i+1}/${retries})...`);
      lastError = error;
      
      // Wait before retrying
      await new Promise(resolve => setTimeout(resolve, delay));
    }
  }
  
  throw lastError;
}

/**
 * Handle Telegram API errors
 */
function handleTelegramError(error) {
  // Specific handling for common Telegram API errors
  if (error.code === 'ETELEGRAM') {
    console.error('Telegram API error:', error.description);
    return `Telegram API Error: ${error.description}`;
  } else if (error.code === 'EFATAL') {
    console.error('Fatal Telegram error:', error.message);
    return 'Connection to Telegram failed. Will retry automatically.';
  } else {
    console.error('Unknown error:', error);
    return 'An error occurred. Please try again later.';
  }
}

module.exports = {
  retryOperation,
  handleTelegramError
};
