// Save every scanned token (address, network, marketcap, timestamp) to radar_a_tg_result
async function saveScannedToken(address, network, groupId = null, firstMcap = null, groupLink = null, usernameCalled = null) {
  // Always insert a new row (track every call)
  const now = Math.floor(Date.now() / 1000);
  const today = new Date().toISOString().slice(0, 10);
  if (!firstMcap) throw new Error('firstMcap (marketcap) is required for first scan!');
  await pool.query(
    'INSERT INTO radar_a_tg_result (group_id, contract_address, first_mcap, first_called_at, date_first_call, group_link, username_called, network, scanned_at) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)',
    [groupId, address, firstMcap, now, today, groupLink, usernameCalled, network, now]
  );
}

// PostgreSQL handler for first call marketcap per group and contract address
const { Pool } = require('pg');
const { parseMarketCap } = require('./numberUtils');

// Build pool config safely. Accept either a full PG_URI or component env vars.
function buildPgConfig() { 
  const uri = process.env.PG_URI;
  if (!uri) {
    throw new Error('PG_URI environment variable is required for database connection. Set PG_URI to your Postgres connection string.');
  }
  return { connectionString: uri };
}

const pool = new Pool(buildPgConfig());

async function ensureTable() {
  try {
    // Ensure both helper tables exist. Use DOUBLE PRECISION for numeric storage of mcaps.
    await pool.query(`CREATE TABLE IF NOT EXISTS first_call_mcap (
      group_id TEXT NOT NULL,
      contract_address TEXT NOT NULL,
      first_mcap DOUBLE PRECISION,
      first_called_at BIGINT NOT NULL,
      PRIMARY KEY (group_id, contract_address)
    )`);

    await pool.query(`CREATE TABLE IF NOT EXISTS radar_a_tg_result (
      group_id TEXT,
      contract_address TEXT NOT NULL,
      first_mcap DOUBLE PRECISION,
      first_called_at BIGINT,
      date_first_call DATE,
      group_link TEXT,
      username_called TEXT,
      network TEXT,
      scanned_at BIGINT
    )`);

    // If older versions of the tables exist with TEXT columns, attempt a migration
    // Strategy: add a new numeric column, parse existing text values in JS and copy
    // them into the numeric column, then drop the old text column and rename.
    // This avoids dangerous SQL-only casts which can fail on formats like '639.6K' or '$639,600'.

    // Helper to migrate a table/column
    async function migrateTextToNumeric(tableName) {
      try {
        const colRes = await pool.query(
          `SELECT data_type FROM information_schema.columns WHERE table_name = $1 AND column_name = 'first_mcap'`,
          [tableName]
        );
        if (!colRes.rows || colRes.rows.length === 0) return; // no column found
        const dataType = colRes.rows[0].data_type;
        if (dataType === 'double precision') return; // already numeric

        console.log(`[firstCallDb] Migrating ${tableName}.first_mcap from ${dataType} -> double precision`);

        // add new numeric column
        await pool.query(`ALTER TABLE ${tableName} ADD COLUMN IF NOT EXISTS first_mcap_num DOUBLE PRECISION`);

        // Read all rows and update numeric column using JS parser
        const rows = await pool.query(`SELECT group_id, contract_address, first_mcap, first_called_at FROM ${tableName}`);
        for (const r of rows.rows) {
          const raw = r.first_mcap;
          if (raw === undefined || raw === null) continue;
          const parsed = parseMarketCap(String(raw));
          if (parsed === null || isNaN(parsed)) continue;
          // Use group_id + contract_address + first_called_at as an identifying tuple
          await pool.query(
            `UPDATE ${tableName} SET first_mcap_num = $1 WHERE contract_address = $2 AND first_called_at = $3 AND (group_id = $4 OR (group_id IS NULL AND $4 IS NULL))`,
            [parsed, r.contract_address, r.first_called_at, r.group_id]
          );
        }

        // Drop old column and rename new
        await pool.query(`ALTER TABLE ${tableName} DROP COLUMN IF EXISTS first_mcap`);
        await pool.query(`ALTER TABLE ${tableName} RENAME COLUMN first_mcap_num TO first_mcap`);
        console.log(`[firstCallDb] Migration completed for ${tableName}`);
      } catch (mErr) {
        console.error(`[firstCallDb] Migration failed for ${tableName}:`, mErr.message);
      }
    }

    // Attempt migration for both tables (if necessary)
    await migrateTextToNumeric('first_call_mcap');
    await migrateTextToNumeric('radar_a_tg_result');
  } catch (err) {
    // Provide a clearer error message for common auth/connection mistakes
    console.error('[firstCallDb] Error ensuring table exists. DB config:', {
      host: process.env.PG_HOST || null,
      port: process.env.PG_PORT || null,
      user: process.env.PG_USER || null,
      database: process.env.PG_DATABASE || null,
      usingUri: !!process.env.PG_URI
    });
    // Rethrow so callers can handle/log the original error stack
    throw err;
  }
}

async function getFirstCall(groupId, contractAddress) {
  await ensureTable();
  const res = await pool.query(
      'SELECT * FROM radar_a_tg_result WHERE group_id = $1 AND contract_address = $2 ORDER BY date_first_call ASC LIMIT 1',
    [groupId, contractAddress]
  );
  return res.rows[0];
}

async function getLatestScan(groupId, contractAddress) {
  await ensureTable();
  const res = await pool.query(
    'SELECT * FROM radar_a_tg_result WHERE group_id = $1 AND contract_address = $2 ORDER BY scanned_at DESC LIMIT 1',
    [groupId, contractAddress]
  );
  return res.rows[0];
}

async function setFirstCall(groupId, contractAddress, mcap, groupLink = null, usernameCalled = null, network = null) {
  // Always insert a new row (track every call)
  const now = Math.floor(Date.now() / 1000);
  const today = new Date().toISOString().slice(0, 10);
  await pool.query(
      'INSERT INTO radar_a_tg_result (group_id, contract_address, first_mcap, first_called_at, date_first_call, group_link, username_called, network, scanned_at) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)',
    [groupId, contractAddress, mcap, now, today, groupLink, usernameCalled, network, now]
  );
}

module.exports = {
  getFirstCall,
  setFirstCall,
  saveScannedToken,
  getLatestScan
};
