const { analyzeTokenSecurity } = require('../services/securityAnalyzer');
// Remove image generator import
// const { generateResultImage } = require('./imageGenerator');

/**
 * Format token analysis in a readable way similar to SafeAnalyzer
 */
async function formatTokenAnalysis(tokenDetails, dexData, network) {
  if (!tokenDetails) {
    return {
      text: "❌ Could not analyze token. Please check the address and try again.",
      image: null
    };
  }

  // Perform security analysis
  const securityAnalysis = await analyzeTokenSecurity(tokenDetails, dexData);
  
  // Safety indicator based on security analysis
  const safetyIndicator = securityAnalysis.emoji;
  
  // Add a text-based security indicator at the beginning for better visibility
  let securityText = '';
  switch(securityAnalysis.status) {
    case 'SAFE':
      securityText = '🟢 SAFE';
      break;
    case 'LIKELY SAFE':
      securityText = '🟢 LIKELY SAFE';
      break;
    case 'CAUTION':
      securityText = '🟡 CAUTION';
      break;
    case 'HIGH RISK':
      securityText = '🔴 HIGH RISK';
      break;
    default:
      securityText = '⚪ UNKNOWN';
  }
  
  // Format market cap dengan suffix yang tepat - mengutamakan tampilan yang sama dengan DexScreener
  const formatMCap = (mcap) => {
    // Debug log to track the value and type
    console.log(`formatMCap received: ${mcap} (type: ${typeof mcap})`);

    // Validasi data market cap
    if (mcap === undefined || mcap === null || mcap === 0) {
      return "Unknown";
    }
    
    try {
      // Ensure we're working with string or number only 
      if (typeof mcap !== 'string' && typeof mcap !== 'number') {
        // Fix the syntax error in this line - correct the template string
        console.log(`Warning: mcap is type ${typeof mcap}, converting to string`);
        mcap = String(mcap);  // Convert to string safely
      }
      
      // Check for string formats with currency symbols/suffixes
      if (typeof mcap === 'string') {
        if (mcap.includes('$') || mcap.includes('K') || mcap.includes('M') || mcap.includes('B')) {
          return mcap.startsWith('$') ? mcap : '$' + mcap;
        }
      }
      
      // Konversi dari string ke number jika diperlukan
      const numericMcap = typeof mcap === 'string' ? parseFloat(mcap) : mcap;
      
      if (isNaN(numericMcap)) {
        return "Unknown";
      }
      
      // Format dengan angka yang tepat seperti DexScreener (dengan 1 desimal)
      if (numericMcap >= 1e9) return `$${(numericMcap / 1e9).toFixed(1)}B`;
      if (numericMcap >= 1e6) return `$${(numericMcap / 1e6).toFixed(1)}M`;
      if (numericMcap >= 1e3) return `$${(numericMcap / 1e3).toFixed(1)}K`;
      return `$${numericMcap.toFixed(2)}`;
    } catch (error) {
      console.error(`Error formatting market cap: ${error.message}`);
      return "Unknown";
    }
  };
  
  // Format number with commas as thousands separator
  // Treat only null/undefined or non-numeric values as Unknown. Allow 0.
  const formatNumber = (num) => {
    if (num === undefined || num === null) return "Unknown";
    const n = Number(num);
    if (isNaN(n)) return "Unknown";

    // Convert to string and use dots as thousands separators
    return n.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ".");
  };
  
  // Ensure we use the correct network from tokenDetails if available
  // This catches cases where network detection refined the network type
  if (tokenDetails && tokenDetails.network) {
    network = tokenDetails.network;
  }
  
  // Network explorer URL
  const getExplorerUrl = () => {
    switch(network) {
      case 'BSC':
        return `https://bscscan.com/token/${tokenDetails.contractAddress}`;
      case 'ETH':
        return `https://etherscan.io/token/${tokenDetails.contractAddress}`;
      case 'SOL':
        return `https://solscan.io/token/${tokenDetails.contractAddress}`;
      default:
        return '#';
    }
  };
  
  // Shortened output for efficiency - keep the most important info
  // and ensure total length doesn't exceed Telegram limits (max ~1000 chars for captions)
  let output = `RadarAnalyzer | ${tokenDetails.symbol || "Unknown"} | \n\n`;
  
  // Add security text prominently at the top
  output += `${securityText}\n\n`;
  
  // Ensure network is displayed correctly in bold
  output += `🔸<b>${network}</b>: <code>${tokenDetails.contractAddress}</code>\n`;

  // Format owner address for better display
  let ownerDisplay = "Unknown";
  if (tokenDetails.owner === 'RENOUNCED') {
    ownerDisplay = '✅ REVOKED';
  } else if (tokenDetails.owner && tokenDetails.owner !== 'Unknown') {
    // Show the actual developer wallet address instead of just "NOT RENOUNCED"
    ownerDisplay = `❌ ${tokenDetails.owner}`;
  }

  output += `├ Owner:    ${ownerDisplay}\n`;
  
  // Conditional showing of transaction data to save space
  if (dexData && dexData.extendedDexData && dexData.extendedDexData.txns && 
      dexData.extendedDexData.txns.h24 && 
      (dexData.extendedDexData.txns.h24.buys > 2 || dexData.extendedDexData.txns.h24.sells > 2)) {
    output += `├ Txns 24h: ${dexData.extendedDexData.txns.h24.buys || 0} buys, ${dexData.extendedDexData.txns.h24.sells || 0} sells\n`;
  }
  
  if (dexData) {
    // Shortened display of DEX ID
    const dexIdDisplay = dexData.dexId;
    
    // Only show LP Ratio if we have reliable data
    if (dexData.liquidityUSD && dexData.marketCap) {
      const lpRatio = (parseFloat(dexData.liquidityUSD) / parseFloat(dexData.marketCap || '1') * 100).toFixed(1);
      output += `├ LP Ratio: ${lpRatio}% in ${dexIdDisplay}.\n`;
    }

    // Make market cap and price display more concise
    let marketCapDisplay = "Unknown";
    let priceDisplay = "Unknown";
    
    try {
      if (dexData.marketCap !== undefined && dexData.marketCap !== null) {
        marketCapDisplay = formatMCap(dexData.marketCap);
      }
      
      if (dexData.price) {
        const priceNum = parseFloat(dexData.price);
        // Determine decimal precision based on value
        const decimals = priceNum < 0.00001 ? 10 : 
                        priceNum < 0.01 ? 6 : 
                        priceNum < 1 ? 4 : 2;
        priceDisplay = `$${priceNum.toFixed(decimals)}`;
      }
      
      // Display market cap and price on separate lines
      output += `├ MCap: ${marketCapDisplay}\n`;
      output += `├ Price: ${priceDisplay}\n`;
      
      // Always display FDV if available (not conditional anymore)
      if (dexData.fdv) {
        const fdvFormatted = formatMCap(dexData.fdv);
        output += `├ FDV: ${fdvFormatted}\n`;
      }

      // Liquidity is important, keep it
      output += `├ Liquid: $${formatNumber(Math.round(dexData.liquidityUSD || 0))}\n`;
      
      // Volume is important for trading
      if (dexData.volume24h && parseFloat(dexData.volume24h) > 0) {
        output += `├ Vol24h: $${formatNumber(Math.round(dexData.volume24h || 0))}\n`;
      }
    } catch (error) {
      console.error(`Error displaying market data: ${error.message}`);
    }
  }
  
  // Tax information is critical
  output += `├ Tax: Buy: ${tokenDetails.taxInfo?.buy || '0.0%'} | Sell: ${tokenDetails.taxInfo?.sell || '0.0%'}\n`;
  output += `├ Age: ${tokenDetails.ageFormatted}\n`;
  
  // Honeypot check is critical for security
  if (securityAnalysis.honeypot && network !== 'SOL') {
    const honeypotStatus = securityAnalysis.honeypot;
    let honeypotEmoji = '❓';
    let honeypotDisplay = 'UNKNOWN';
    
    if (honeypotStatus.isHoneypot) {
      honeypotEmoji = '⚠️';
      honeypotDisplay = 'YES';
    } else if (honeypotStatus.confidence === 'Safe' || honeypotStatus.confidence === 'Low Risk') {
      honeypotEmoji = '✅';
      honeypotDisplay = 'NO';
    }
    
    output += `├ Honeypot: ${honeypotEmoji} ${honeypotDisplay}\n`;
  }
  
  // Move key security features right after honeypot check (without the security heading)
  // Use emojis for Yes/No indicators with consistent formatting - now with square brackets
  const ownershipRenounced = tokenDetails.owner === 'RENOUNCED';
  const taxModifiable = tokenDetails.taxModifiable === true;
  const hiddenOwner = securityAnalysis.contractFeatures?.hasHiddenOwner === true;
  const cantSellAll = securityAnalysis.contractFeatures?.cantSellAll === true;
  
  // Add DEX Paid indicator back (skip for Solana)
  if (dexData && dexData.isPaid !== undefined && network !== 'SOL') {
    const isPaid = dexData.isPaid === true;
    output += `├ DEX Paid: ${isPaid ? '✅ Yes' : '❌ No'}\n`;
  }
  
  // Add contract verification status indicator (skip for Solana and ETH)
  if (network !== 'SOL' && network !== 'ETH') {
    const isVerified = dexData?.isVerified === true || tokenDetails.verified === true;
    output += `├ Verified Contract: ${isVerified ? '✅ Yes' : '❌ No'}\n`;
  }
  
  // Add Solana-specific indicators
  const isSolana = network === 'SOL';
  if (isSolana) {
    const mintable = tokenDetails.mintable === true;
    const freezable = tokenDetails.freezable === true;
    
    // Show SolSniffer score if available
    if (tokenDetails.solSnifferScore) {
      output += `├ SolSniffer Score: ${tokenDetails.solSnifferScore}/100\n`;
    }
    
    // Add Solana-specific security indicators
    output += `├ Mintable Token: ${mintable ? '❌ Yes' : '✅ No'}\n`;
    output += `├ Can Freeze Accounts: ${freezable ? '❌ Yes' : '✅ No'}\n`;
  } else {
    // For non-Solana tokens, show the regular EVM security indicators
    output += `├ Hidden Owner: ${hiddenOwner ? '❌ Yes' : '✅ No'}\n`;
    output += `├ Can't Sell All: ${cantSellAll ? '❌ Yes' : '✅ No'}\n`;
    output += `├ Modifiable Tax: ${taxModifiable ? '❌ Yes' : '✅ No'}\n`;
  }
  
  // Socials are important for community verification
  let socialsText = '';
  if (tokenDetails.socials) {
    const links = [];
    
    if (tokenDetails.socials.telegram) {
      links.push(`<a href="${tokenDetails.socials.telegram}">Telegram</a>`);
    }
    
    if (tokenDetails.socials.twitter) {
      links.push(`<a href="${tokenDetails.socials.twitter}">Twitter</a>`);
    }
    
    if (tokenDetails.socials.website) {
      links.push(`<a href="${tokenDetails.socials.website}">Website</a>`);
    }
    
    if (links.length > 0) {
      socialsText = links.join(' | ');
    }
  }

  // Add social links if they exist
  if (socialsText) {
    output += `├ Socials: ${socialsText}\n\n`;
  } else {
    output += `├ Socials: Not found\n\n`;
  }
  
  // Only include critical warnings
  if (securityAnalysis.warnings && securityAnalysis.warnings.length > 0) {
    const criticalWarnings = securityAnalysis.warnings.filter(warning => 
      warning.includes('honeypot') || 
      warning.includes('scam') || 
      warning.includes('blacklist') ||
      warning.includes('high tax'));
      
    if (criticalWarnings.length > 0) {
      output += `⚠️ Security Warnings:\n${criticalWarnings.slice(0, 2).join('\n')}\n\n`;
    }
  }
  
  // Add holder count if available
  const holderCount = dexData?.holderCount || tokenDetails.holderCount || 'Unknown';
  if (holderCount !== 'Unknown') {
    output += `├ 👥 Holders: ${holderCount}\n`;
  }
  
  // Disclaimer and links section - essential
  output += `\n🎯 ${tokenDetails.symbol} | DYOR, NFA\n\n`;
  
  // Add trading bot links in compact form
  const maestroLink = `https://t.me/maestro?start=${tokenDetails.contractAddress}-abxprt`;
  const sigmaLink = `https://t.me/Sigma_buyBot?start=x780283668-${tokenDetails.contractAddress}`;
  
  output += `🤖 <a href="${maestroLink}">Maestro</a> | <a href="${sigmaLink}">Sigma</a> | Powered by RadarAnalyzer.io`;
  
  return {
    text: output,
    securityStatus: securityAnalysis.status,
    contractAddress: tokenDetails.contractAddress,
    includeAd: true
  };
}

module.exports = {
  formatTokenAnalysis
};
