const axios = require('axios');

/**
 * Check if a token is a honeypot using external APIs
 * @param {string} address - Token contract address
 * @param {string} network - Blockchain network
 * @returns {Promise<Object>} Honeypot analysis result
 */
async function checkHoneypotStatus(address, network) {
  try {
    console.log(`Checking honeypot status for ${address} on ${network}`);
    
    // Default response structure
    const result = {
      isHoneypot: false,
      confidence: 'Unknown',
      warning: '',
      buyTax: null,
      sellTax: null,
      errors: [],
      source: null
    };
    
    // First try GoPlusSecurity API
    try {
      const goPlusResult = await checkWithGoPlusSecurity(address, network);
      if (goPlusResult) {
        console.log(`Got honeypot data from GoPlus: ${JSON.stringify(goPlusResult)}`);
        return {
          ...result,
          ...goPlusResult,
          source: 'GoPlus Security'
        };
      }
    } catch (error) {
      console.log(`Error with GoPlus API: ${error.message}`);
      result.errors.push(`GoPlus: ${error.message}`);
    }
    
    // Then try Honeypot.is API
    try {
      const honeypotIsResult = await checkWithHoneypotIs(address, network);
      if (honeypotIsResult) {
        console.log(`Got honeypot data from Honeypot.is: ${JSON.stringify(honeypotIsResult)}`);
        return {
          ...result,
          ...honeypotIsResult,
          source: 'Honeypot.is'
        };
      }
    } catch (error) {
      console.log(`Error with Honeypot.is API: ${error.message}`);
      result.errors.push(`Honeypot.is: ${error.message}`);
    }

    // Try HoneypotScanner API
    try {
      const scannerResult = await checkWithHoneypotScanner(address, network);
      if (scannerResult) {
        console.log(`Got honeypot data from HoneypotScanner: ${JSON.stringify(scannerResult)}`);
        return {
          ...result,
          ...scannerResult,
          source: 'HoneypotScanner'
        };
      }
    } catch (error) {
      console.log(`Error with HoneypotScanner API: ${error.message}`);
      result.errors.push(`HoneypotScanner: ${error.message}`);
    }
    
    // If all external APIs failed, return the default structure
    return result;
  } catch (error) {
    console.error(`Error in honeypot detection:`, error);
    return {
      isHoneypot: false,
      confidence: 'Error',
      warning: `Error checking honeypot: ${error.message}`,
      errors: [error.message]
    };
  }
}

/**
 * Check honeypot status using GoPlusSecurity API
 */
async function checkWithGoPlusSecurity(address, network) {
  // Map our network names to GoPlus network IDs
  const networkMapping = {
    'BSC': '56',
    'ETH': '1',
    'POLYGON': '137',
    'AVALANCHE': '43114'
  };
  
  const chainId = networkMapping[network.toUpperCase()];
  if (!chainId) {
    throw new Error(`Network ${network} not supported by GoPlus API`);
  }
  
  const apiUrl = `https://api.gopluslabs.io/api/v1/token_security/${chainId}/${address}`;
  const response = await axios.get(apiUrl);
  
  if (response.data && response.data.code === 1 && response.data.result) {
    const securityData = response.data.result;
    
    // Check honeypot indicators
    const isHoneypot = 
      securityData.is_honeypot === '1' || 
      securityData.cannot_sell === '1' ||
      securityData.trading_cooldown === '1';
    
    // Extract tax info if available
    const buyTax = securityData.buy_tax ? parseFloat(securityData.buy_tax) * 100 : null;
    const sellTax = securityData.sell_tax ? parseFloat(securityData.sell_tax) * 100 : null;
    
    // Generate warnings
    const warnings = [];
    if (securityData.is_honeypot === '1') warnings.push('Identified as honeypot');
    if (securityData.cannot_sell === '1') warnings.push('Cannot sell token');
    if (securityData.trading_cooldown === '1') warnings.push('Has trading cooldown');
    if (securityData.is_anti_whale === '1') warnings.push('Has anti-whale mechanism');
    if (securityData.is_blacklisted === '1') warnings.push('Address is blacklisted');
    if (securityData.is_in_dex === '0') warnings.push('Not available in DEX');
    if (securityData.is_mintable === '1') warnings.push('Token is mintable');
    
    return {
      isHoneypot: isHoneypot,
      confidence: isHoneypot ? 'High' : (warnings.length > 0 ? 'Medium' : 'Safe'),
      warning: warnings.join(', '),
      buyTax,
      sellTax
    };
  }
  
  return null;
}

/**
 * Check honeypot status using Honeypot.is API
 */
async function checkWithHoneypotIs(address, network) {
  // Map our network names to honeypot.is network names
  const networkMapping = {
    'BSC': 'bsc',
    'ETH': 'eth',
    'POLYGON': 'polygon',
    'ARBITRUM': 'arbitrum',
    'OPTIMISM': 'optimism',
    'BASE': 'base'
  };
  
  const chainName = networkMapping[network.toUpperCase()];
  if (!chainName) {
    throw new Error(`Network ${network} not supported by Honeypot.is API`);
  }
  
  const apiUrl = `https://api.honeypot.is/v2/IsHoneypot?address=${address}&chainID=${chainName}`;
  const response = await axios.get(apiUrl, { 
    timeout: 15000,
    headers: {
      'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
    }
  });
  
  if (response.data) {
    const honeypotData = response.data;
    
    // Check honeypot status
    const isHoneypot = honeypotData.honeypotResult?.isHoneypot === true;
    
    // Extract tax info
    const buyTax = honeypotData.simulationResult?.buyTax ? 
      parseFloat(honeypotData.simulationResult.buyTax) * 100 : null;
    const sellTax = honeypotData.simulationResult?.sellTax ? 
      parseFloat(honeypotData.simulationResult.sellTax) * 100 : null;
    
    // Generate warning message
    let warning = '';
    if (honeypotData.honeypotResult?.honeypotReason) {
      warning = honeypotData.honeypotResult.honeypotReason;
    }
    
    return {
      isHoneypot: isHoneypot,
      confidence: isHoneypot ? 'High' : 'Safe',
      warning: warning,
      buyTax,
      sellTax
    };
  }
  
  return null;
}

/**
 * Check honeypot status using HoneypotScanner API
 */
async function checkWithHoneypotScanner(address, network) {
  // Currently only supports BSC
  if (network.toUpperCase() !== 'BSC') {
    throw new Error(`Network ${network} not supported by HoneypotScanner API`);
  }
  
  const apiUrl = `https://honeypot.api.rugdoc.io/api/honeypotStatus.js?address=${address}`;
  
  try {
    const response = await axios.get(apiUrl, { timeout: 8000 });
    const data = response.data;
    
    const isHoneypot = !data.status || data.status === 'PROBABLE_HONEYPOT' || data.status === 'CONFIRMED_HONEYPOT';
    
    // Generate confidence level
    let confidence = 'Unknown';
    if (data.status === 'CONFIRMED_HONEYPOT') confidence = 'High';
    else if (data.status === 'PROBABLE_HONEYPOT') confidence = 'Medium';
    else if (data.status === 'OK' || data.status === 'LOW_LIQUIDITY') confidence = 'Safe';
    
    // Extract warnings
    let warning = data.message || '';
    if (data.status === 'LOW_LIQUIDITY') {
      warning = 'Low liquidity, not a honeypot';
    }
    
    return {
      isHoneypot: isHoneypot,
      confidence: confidence,
      warning: warning,
      buyTax: null, // API doesn't provide tax info
      sellTax: null
    };
  } catch (error) {
    if (error.response && error.response.status === 503) {
      // When the API returns 503, it's often because it can't analyze the token
      return {
        isHoneypot: false,
        confidence: 'Unknown',
        warning: 'Token analysis unavailable',
        buyTax: null,
        sellTax: null
      };
    }
    throw error;
  }
}

module.exports = {
  checkHoneypotStatus
};
