const axios = require("axios");

// Simple in-memory caches (process-lifetime). Configurable via env vars.
const SYMBOL_TTL_MS = Number(process.env.SYMBOL_CACHE_TTL_MS) || 60 * 60 * 1000; // 1 hour
const WALLET_TTL_MS = Number(process.env.WALLET_CACHE_TTL_MS) || 5 * 60 * 1000; // 5 minutes
const CACHE = {
  symbols: new Map(), // mint -> { value: symbol|null, expires: ms }
  walletAssets: new Map(), // owner -> { value: {...}, expires }
};

// Token list cache (fallback to solana token-list on GitHub)
const TOKENLIST_CACHE = { data: null, expires: 0 };

/**
 * Normalize a single Helius asset item into a standard fungible-token shape.
 * Returns null for non-fungible / unsupported items.
 * Standard shape: { mint, total, decimals, symbol, name, accounts: [{ address, amount }] }
 */
function normalizeHeliusAsset(item, owner) {
  if (!item) return null;
  const iface = String(item.interface || '').toLowerCase();

  // reject obvious NFTs
  if (iface.includes('nft') || iface.includes('nonfungible')) return null;

  // try to locate mint identifier
  const mint = item.id || item.mint || item.token || item.address || null;
  if (!mint) return null;

  // decimals: try multiple places
  const decimalsRaw = item.token_info?.decimals ?? item.token_info?.token?.decimals ?? item.decimals ?? item.content?.metadata?.decimals ?? 0;
  const decimals = Number.isFinite(Number(decimalsRaw)) ? parseInt(decimalsRaw, 10) : 0;

  // symbol/name: prefer content.metadata, then token_info.token, then token_info
  const name = item.content?.metadata?.name ?? item.token_info?.token?.name ?? item.token_info?.name ?? item.metadata?.name ?? null;

  // Try multiple possible locations for symbol (Helius shapes vary). Trim strings.
  const symbolCandidates = [
    item.content?.metadata?.symbol,
    item.metadata?.symbol,
    item.token_info?.token?.symbol,
    item.token_info?.symbol,
    item.token_info?.token?.metadata?.symbol,
    item.token_info?.token?.meta?.symbol,
    item.token_info?.token?.data?.symbol,
    item.token_info?.token?.extensions?.symbol
  ];
  let symbol = null;
  for (const s of symbolCandidates) {
    if (s != null && s !== '') { symbol = String(s).trim(); break; }
  }

  // balance: try various fields. Many Helius responses provide `token_info.balance` in base units.
  let rawBalance = null;
  if (item.token_info && item.token_info.balance != null) rawBalance = item.token_info.balance;
  else if (item.token_info && item.token_info.tokenAmount && item.token_info.tokenAmount.amount != null) rawBalance = item.token_info.tokenAmount.amount;
  else if (item.amount != null) rawBalance = item.amount;
  else if (item.token_amount != null) rawBalance = item.token_amount;

  let total = 0;
  if (rawBalance != null) {
    const nb = Number(rawBalance);
    if (!isNaN(nb)) {
      // If nb appears to already be UI amount (small), keep it; otherwise divide by 10^decimals
      total = Math.abs(nb) < 1 ? nb : nb / Math.pow(10, decimals || 0);
    }
  }

  // Some Helius shapes include token_info.token?.uiAmount or token_info.uiAmount
  if ((!total || total === 0) && item.token_info?.token?.uiAmount != null) total = Number(item.token_info.token.uiAmount) || total;
  if ((!total || total === 0) && item.token_info?.uiAmount != null) total = Number(item.token_info.uiAmount) || total;

  // If still zero but token_info exists and interface suggests fungible, treat as zero (don't include)
  if (!total || Number(total) === 0) return null;

  return {
    mint,
    total: Number(total),
    decimals: decimals || 0,
    symbol: symbol || null,
    name: name || null,
    token_name: name || null,
    accounts: [{ address: owner, amount: Number(total) }]
  };
}

// 🔹 Ganti ini dengan API Key kamu dari https://www.helius.dev  ||"207115c7-67ef-48e8-860c-ae20fbfe1b37"
const API_KEY = process.env.HELIUS_API_KEY;

// 🔹 Endpoint Helius RPC
const RPC_URL = `https://mainnet.helius-rpc.com/?api-key=${API_KEY}`;

// 🔹 Ganti ini dengan mint address token yang ingin kamu analisa
const MINT_ADDRESS = "2iKoxXwc9GPKUQWth2JcdWJXc7rV5Sp4ErBjQSr2pump"; // contoh: USDC

// 🔁 Fungsi helper dengan retry otomatis (tahan rate limit)
async function safePost(data, retries = 3) {
  for (let i = 0; i < retries; i++) {
    try {
      return await axios.post(RPC_URL, data);
    } catch (err) {
      if (err.response?.status === 429) {
        console.log(`⚠️ Rate limited, retrying (${i + 1}/${retries})...`);
        // Gunakan backoff eksponensial sederhana
        await new Promise(r => setTimeout(r, 1000 * Math.pow(2, i)));
      } else {
        // Tampilkan error yang lebih detail
        console.error("Axios Error:", err.response?.data || err.message);
        throw err;
      }
    }
  }
  throw new Error("Failed after retries");
}

async function getTopHolders(mint, topN = 10) {
  try {
    console.time("getTopHolders");
    console.log(`🔍 Menganalisa top ${topN} holder untuk mint: ${mint}`);

    const TOKEN_PROGRAM_ID = "TokenkegQfeZyiNwAJbNbGKPFXCWuBvf9Ss623VQ5DA";
    const MAX_ACCOUNTS = 20000; // batas aman default

    let allAccounts = null;
    try {
      console.log("🚀 Mencoba Strategi 1: getProgramAccounts (Akurat)...");
      const progRes = await safePost({
        jsonrpc: "2.0",
        id: 1,
        method: "getProgramAccounts",
        params: [
          TOKEN_PROGRAM_ID,
          {
            encoding: "jsonParsed",
            filters: [
              // 💡 PERBAIKAN 1: Tambahkan dataSize filter
              // Ini sangat penting untuk memfilter HANYA token account
              { dataSize: 165 },
              // Filter berdasarkan mint address (offset 0)
              { memcmp: { offset: 0, bytes: mint } },
            ],
          },
        ],
      });

      allAccounts = progRes.data.result;
      if (!Array.isArray(allAccounts)) allAccounts = [];
      console.log(`✅ getProgramAccounts berhasil: ${allAccounts.length} token accounts ditemukan.`);

      if (allAccounts.length > MAX_ACCOUNTS) {
        console.warn(`⚠️ getProgramAccounts mengembalikan ${allAccounts.length} akun, membatasi ke ${MAX_ACCOUNTS} pertama untuk performa.`);
        allAccounts = allAccounts.slice(0, MAX_ACCOUNTS);
      }
    } catch (e) {
      // Jika getProgramAccounts gagal, fallback
      console.warn(`⚠️ Strategi 1 gagal — fallback ke Strategi 2: getTokenLargestAccounts...`, e.message);
      const res = await safePost({
        jsonrpc: "2.0",
        id: 1,
        method: "getTokenLargestAccounts",
        params: [mint],
      });

      if (!res?.data?.result?.value) {
        throw new Error('Invalid response from getTokenLargestAccounts');
      }

      // Ambil top 20 (default dari getTokenLargestAccounts)
      const sample = res.data.result.value;
      console.log(`✅ getTokenLargestAccounts berhasil: ${sample.length} akun terbesar ditemukan.`);

      // batch getMultipleAccounts untuk semua sample addresses
      const addresses = sample.map(a => a.address);
      let accountsInfo = null;
      try {
        console.log(`batch-fetching ${addresses.length} account details...`);
        const batchRes = await safePost({
          jsonrpc: "2.0",
          id: 1,
          method: "getMultipleAccounts",
          params: [addresses, { encoding: "jsonParsed" }],
        });
        accountsInfo = batchRes.data.result.value;
      } catch (err2) {
        // Fallback jika getMultipleAccounts gagal (jarang terjadi)
        console.warn("⚠️ Batch getMultipleAccounts gagal, mencoba per-account:", err2.message);
        accountsInfo = [];
        for (const addr of addresses) {
          const accountRes = await safePost({
            jsonrpc: "2.0",
            id: 1,
            method: "getAccountInfo",
            params: [addr, { encoding: "jsonParsed" }],
          });
          accountsInfo.push(accountRes.data.result?.value ?? null);
          await new Promise(r => setTimeout(r, 100)); // rate limit kecil
        }
      }

      // 💡 PERBAIKAN 2: Ubah cara membangun 'allAccounts'
      // accountsInfo[i] SUDAH merupakan objek akun penuh
      allAccounts = sample.map((s, i) => ({
        pubkey: s.address,
        account: accountsInfo[i], // Langsung tetapkan objek akun
      }));
    }

    // --- Parsing dan Agregasi (Logika ini sudah benar) ---

    console.log("🔄 Mem-parsing dan meng-agregasi data akun...");
    const perAccount = [];
    for (const item of allAccounts) {
      if (!item || !item.account) continue; // Skip jika akun null (mungkin ditutup)

      const parsed = item.account?.data?.parsed?.info;
      if (!parsed) continue; // skip non-parsed entries

      const owner = parsed.owner ?? "Unknown";
      const tokenAmount = parsed.tokenAmount ?? {};
      
      // Ambil uiAmount jika ada, jika tidak hitung manual
      const amount = typeof tokenAmount.uiAmount === 'number'
        ? tokenAmount.uiAmount
        : (Number(tokenAmount.amount ?? 0) / Math.pow(10, tokenAmount.decimals ?? 6));

      if (amount > 0) { // Hanya proses akun dengan saldo
        perAccount.push({ tokenAccount: item.pubkey, owner, amount });
      }
    }

    // AGREGASI per owner
    const ownerMap = new Map();
    for (const a of perAccount) {
      const key = a.owner;
      const cur = ownerMap.get(key) || { owner: key, total: 0, accounts: [] };
      cur.total += a.amount;
      cur.accounts.push({ tokenAccount: a.tokenAccount, amount: a.amount });
      ownerMap.set(key, cur);
    }

    const aggregated = Array.from(ownerMap.values())
      .sort((x, y) => y.total - x.total)
      .slice(0, topN);

    // Hitung persentase berdasarkan total yang diagregasi (bukan hanya topN)
    const totalInAggregated = Array.from(ownerMap.values()).reduce((s, v) => s + v.total, 0);

    const finalResult = aggregated.map((item, i) => ({
      rank: i + 1,
      owner: item.owner,
      total: item.total,
      percent: totalInAggregated ? (item.total / totalInAggregated) * 100 : 0,
      accounts: item.accounts,
    }));


    console.timeEnd("getTopHolders");

    // --- Output --- 
    for (const a of finalResult) {
      const short = `${a.owner.slice(0, 4)}...${a.owner.slice(-4)}`;
      const pct = a.percent.toFixed(2) + '%';
      const emoji = a.rank === 1 ? '🥇' : (a.rank === 2 ? '🥈' : (a.rank === 3 ? '🥉' : '🔸')); 
    }

    // console.log('\nTop holders (aggregated by owner):');
    // console.table(finalResult.map(a => ({
    //   rank: a.rank,
    //   owner: a.owner,
    //   total: a.total,
    //   percent: a.percent.toFixed(2) + '%',
    //   num_accounts: a.accounts.length
    // })));

    if (finalResult.length > 0) {
     // console.log(`\nDetail accounts for Top Holder #${finalResult[0].rank} (${finalResult[0].owner}):`);
      // 💡 PERBAIKAN 3: Menghapus blok console.table duplikat
      // console.table(finalResult[0].accounts.sort((a,b) => b.amount - a.amount));
    }
    
    // Return an object (JSON-like) so callers can parse programmatically
    return { items: finalResult };
  } catch (err) {
    const message = err.response?.data?.error?.message || err.message;
    console.error("❌ Error:", message);
    // Return error in JSON-like structure for programmatic callers
    return { items: [], error: message };
  }
}

// Jalankan (boleh passing mint dan topN via command-line)
if (require.main === module) {
  const argvMint = process.argv[2];
  const argvTop = Number(process.argv[3]) || 10;
  const mintToUse = argvMint || MINT_ADDRESS;

  (async () => {
    const res = await getTopHolders(mintToUse, argvTop);
    // If CLI, pretty print JSON
    console.log('\nJSON output:');
    console.log(JSON.stringify(res, null, 2));
  })().catch(e => console.error('Error running getTopHolders:', e && e.message));
} 
module.exports = { getTopHolders };

// Helpers for Telegram formatting
function shortAddr(addr) {
  if (!addr || typeof addr !== 'string') return addr || '';
  return `${addr.slice(0,4)}...${addr.slice(-4)}`;
}

function shortNumber(n) {
  if (n === undefined || n === null) return '0';
  const num = Number(n);
  if (isNaN(num)) return String(n);
  if (Math.abs(num) >= 1e9) return (num / 1e9).toFixed(2) + 'B';
  if (Math.abs(num) >= 1e6) return (num / 1e6).toFixed(2) + 'M';
  if (Math.abs(num) >= 1e3) return (num / 1e3).toFixed(2) + 'K';
  return num.toLocaleString();
}

/**
 * Build Telegram HTML message for top holders.
 * items: array of { rank, address, amount, percent, raw }
 * opts: { titleEmoji, titleName, tokenSymbol }
 */
function buildTelegramMessage(mint, items = [], opts = {}) {
  const titleEmoji = opts.titleEmoji || '💎';
  const titleName = opts.titleName || (mint ? (mint.slice(0,8)) : 'token');

  let lines = [];
  lines.push(`<b>${titleEmoji} Top Wallets for ${escapeHtml(titleName)}</b>`);
  lines.push('');

  for (const it of items.slice(0, 10)) {
    const rank = it.rank || '';
    const addr = it.address || it.owner || (it.raw && it.raw.owner) || '';
    const amount = it.amount || it.total || (it.raw && it.raw.amount) || 0;
    const percent = typeof it.percent === 'number' ? it.percent : (it.raw && it.raw.percent) || null;

  // pick emoji based on heuristics (owner type or USD tier)
  // Determine token amount (prefer explicit total when present)
  const tokenAmount = (it.total || amount || 0);
  const usdPrice = (opts && typeof opts.tokenPriceUsd === 'number') ? opts.tokenPriceUsd : null;
  const usdValue = usdPrice !== null ? (Number(tokenAmount || 0) * usdPrice) : null;
  // Pass tokenAmount (the 'total' from Helius) into selectEmoji so emoji selection is based on that value
  const emoji = selectEmoji(it, tokenAmount, usdValue, rank);
    const link = `https://solscan.io/account/${encodeURIComponent(addr)}`;
    const short = shortAddr(addr);
    const pctText = percent !== null && percent !== undefined ? ` [${percent.toFixed(2)}%]` : '';

  // First line: rank, linked short address, percent and emoji
  lines.push(`#${rank} <a href="${link}">${escapeHtml(short)}</a>${pctText} ${emoji}`);

  // Display wallet token contents (preferred) or fallback to the queried token total.
  // Format for each shown token: "- {Token_Name} {Token_balance}" (default: top 3 tokens).
  const wa = it._walletAssets;
  const maxShow = (typeof opts.maxTokensShown === 'number') ? opts.maxTokensShown : 3;
  if (wa && Array.isArray(wa.tokens) && wa.tokens.length > 0) {
    const tokensToShow = (maxShow > 0) ? wa.tokens.slice(0, maxShow) : wa.tokens;
    for (const t of tokensToShow) {
      const label = t.symbol || shortMint(t.mint) || 'TOKEN';
      const decimals = (t.decimals !== undefined && t.decimals !== null) ? t.decimals : (opts.tokenDecimals || 6);
      const amt = formatTokenAmount(t.total, decimals);
      lines.push(`- ${escapeHtml(String(label))} ${escapeHtml(String(amt))}`);
    }
  } else {
    // Fallback: show the queried token symbol and the aggregated amount (if available)
    const amountText = formatTokenAmount(tokenAmount, opts.tokenDecimals || 6);
    //lines.push(`L ${escapeHtml(String(opts.tokenSymbol || 'TOKEN'))} ${amountText}`);
    //lines.push(`L ${fetchTokenSymbol(t.mint)} ${amountText}`);
  }
    // blank separator
    lines.push('');
  }

  // footer: clickable link to full holders page
  const holdersLink = `https://solscan.io/token/${encodeURIComponent(mint)}/holders`;
  lines.push(`Powered By <a href="https://www.radaranalyzer.io/">RadarAnalyzer.io</a>`);

  // Join with newlines and return
  return lines.join('\n');
}

function escapeHtml(str) {
  return String(str)
    .replace(/&/g, '&amp;')
    .replace(/</g, '&lt;')
    .replace(/>/g, '&gt;')
    .replace(/"/g, '&quot;');
}

// Export formatter (add to existing exports)
// Emoji mapping defaults (based on your provided table). You can tweak these constants.
const EMOJI_MAP = {
  LP: '💧',
  EXCHANGE: '🏦',
  LP_BURNED: '🔥',
  LP_LOCKED: '🔒',
  DEPLOYER: '👷',
  NEW_WALLET: '🌱',
  KNOWN_WALLET: '👀',
  KNOWN_BOT: '🤖',
  TIER_LT_100: '🦐',
  TIER_100_1K: '🍤',
  TIER_1K_5K: '🐟',
  TIER_5K_50K: '🐬',
  TIER_50K_200K: '🐋',
  TIER_GT_200K: '🐳'
};

function selectEmoji(item, tokenAmount, usdValue, rank) {
  // Priority: explicit tags in raw -> tokenAmount tier (Helius 'total') -> usd tier -> rank-based medals -> default
  if (item && item.raw) {
    const raw = item.raw;
    if (raw.isLP) return EMOJI_MAP.LP;
    if (raw.isExchange) return EMOJI_MAP.EXCHANGE;
    if (raw.isLpBurned) return EMOJI_MAP.LP_BURNED;
    if (raw.isLpLocked) return EMOJI_MAP.LP_LOCKED;
    if (raw.isDeployer) return EMOJI_MAP.DEPLOYER;
    if (raw.isNew) return EMOJI_MAP.NEW_WALLET;
    if (raw.isKnown) return EMOJI_MAP.KNOWN_WALLET;
    if (raw.isBot) return EMOJI_MAP.KNOWN_BOT;
  }

  // If Helius provided a numeric token total, use tokenAmount tiers (these thresholds are token-unit based).
  if (typeof tokenAmount === 'number' && !isNaN(tokenAmount)) {
    // Example thresholds (token units) - adjust if you prefer different cutoffs.
    if (tokenAmount < 100) return EMOJI_MAP.TIER_LT_100;
    if (tokenAmount < 1000) return EMOJI_MAP.TIER_100_1K;
    if (tokenAmount < 5000) return EMOJI_MAP.TIER_1K_5K;
    if (tokenAmount < 50000) return EMOJI_MAP.TIER_5K_50K;
    if (tokenAmount < 200000) return EMOJI_MAP.TIER_50K_200K;
    return EMOJI_MAP.TIER_GT_200K;
  }

  // Fallback to USD-based tiers if tokenAmount not available
  if (typeof usdValue === 'number' && !isNaN(usdValue)) {
    if (usdValue < 100) return EMOJI_MAP.TIER_LT_100;
    if (usdValue < 1000) return EMOJI_MAP.TIER_100_1K;
    if (usdValue < 5000) return EMOJI_MAP.TIER_1K_5K;
    if (usdValue < 50000) return EMOJI_MAP.TIER_5K_50K;
    if (usdValue < 200000) return EMOJI_MAP.TIER_50K_200K;
    return EMOJI_MAP.TIER_GT_200K;
  }

  // Fallback to medal for top ranks
  if (rank === 1) return '🥇';
  if (rank === 2) return '🥈';
  if (rank === 3) return '🥉';
  return '🔸';
}

// (exports finalized at end of file)

// Try to fetch token USD price (Birdeye public API) as a best-effort helper.
async function fetchTokenPriceUsd(mint) {
  // Disabled: price lookups via external services are not used. Return null.
  return null;
}

// Try to fetch token symbol/metadata (best-effort). Uses the Solana token-list (GitHub) only.
async function fetchTokenSymbol(mint) {
  try {
    const cached = CACHE.symbols.get(mint);
    const now = Date.now();
    if (cached && cached.expires > now) return cached.value;

    let val = null;
    try {
      const now2 = Date.now();
      if (!TOKENLIST_CACHE.data || TOKENLIST_CACHE.expires <= now2) {
        const url = 'https://raw.githubusercontent.com/solana-labs/token-list/main/src/tokens/solana.tokenlist.json';
        const res = await axios.get(url, { timeout: 10000 });
        TOKENLIST_CACHE.data = res.data && res.data.tokens ? res.data.tokens : null;
        TOKENLIST_CACHE.expires = Date.now() + (12 * 60 * 60 * 1000);
      }
      if (TOKENLIST_CACHE.data && Array.isArray(TOKENLIST_CACHE.data)) {
        const found = TOKENLIST_CACHE.data.find(t => (t.address || '').toLowerCase() === (mint || '').toLowerCase());
        if (found && found.symbol) val = String(found.symbol).trim();
      }
    } catch (e) {
      val = null;
    }

    CACHE.symbols.set(mint, { value: val, expires: now + SYMBOL_TTL_MS });
    return val;
  } catch (e) {
    const now = Date.now();
    CACHE.symbols.set(mint, { value: null, expires: now + Math.min(SYMBOL_TTL_MS, 60 * 1000) });
    return null;
  }
}

// Async wrapper: fetch price (if not provided) then build message
async function buildTelegramMessageAsync(mint, items = [], opts = {}) {
  if (!opts.tokenPriceUsd) {
    const p = await fetchTokenPriceUsd(mint);
    if (p) opts.tokenPriceUsd = p;
  }
  // Enrich each item with wallet assets (best-effort). This allows the sync builder to render holdings.
  try {
    await Promise.all((items || []).map(async it => {
      try {
        const owner = it.owner || it.address || (it.raw && it.raw.owner);
        if (!owner) return;
        // Use getWalletTotalAssets as requested (includes USD valuations)
        const wa = await getWalletTotalAssets(owner, { maxAccounts: 2000 });
        // try to fetch token symbols for each mint (best-effort, cached)
        if (wa && Array.isArray(wa.tokens) && wa.tokens.length > 0) {
          await Promise.all(wa.tokens.map(async t => {
            try {
              const sym = await fetchTokenSymbol(t.mint);
              t.symbol = sym || t.symbol || null;
            } catch (e) {
              t.symbol = t.symbol || null;
            }
          }));
        }
        it._walletAssets = wa;
      } catch (e) {
        // ignore per-item errors
      }
    }));
  } catch (e) {
    // ignore overall enrichment errors
  }
  return buildTelegramMessage(mint, items, opts);
}

function shortMint(mint) {
  if (!mint || typeof mint !== 'string') return '';
  if (mint.length <= 12) return mint;
  return `${mint.slice(0,4)}...${mint.slice(-4)}`;
}

function formatTokenAmount(amount, decimals = 6) {
  const num = Number(amount || 0);
  if (isNaN(num)) return String(amount);
  // For very large numbers use shortNumber
  if (Math.abs(num) >= 1000) return shortNumber(num);
  // Show up to `decimals` decimal places but trim trailing zeros
  const places = Math.min(Math.max(decimals, 0), 6);
  return Number(num.toFixed(places)).toString();
}

module.exports = { getTopHolders, buildTelegramMessage, buildTelegramMessageAsync, selectEmoji, EMOJI_MAP };

/**
 * Helius paginated getAssetsByOwner usage (user-provided).
 * Calls Helius RPC method `getAssetsByOwner` with pagination and converts the result
 * into the module's expected shape: { owner, tokens: [{ mint, total, decimals, accounts }], totalTokenAccounts }
 */
async function getAssetsByOwner(owner, opts = {}) {
  if (!owner || typeof owner !== 'string') {
    return { owner, tokens: [], totalTokenAccounts: 0, error: 'invalid owner' };
  }

  // cache check
  const now = Date.now();
  const cached = CACHE.walletAssets.get(owner);
  if (cached && cached.expires > now) return cached.value;

  console.log(`🔍 Mengambil aset untuk wallet: ${owner}...`);
  console.time(`getWalletAssets:${owner}`);

  let allAssets = [];
  let page = 1;
  const limit = opts.limit || 1000;
  let hasMore = true;

  try {
    while (hasMore) {
      const payload = {
        jsonrpc: '2.0',
        id: `getAssetsByOwner-${owner}-${page}`,
        method: 'getAssetsByOwner',
        params: {
          ownerAddress: owner,
          page: page,
          limit: limit,
        },
      };

      const response = await safePost(payload);
      const result = response && response.data ? response.data.result : null;

      if (result && Array.isArray(result.items) && result.items.length > 0) {
        allAssets.push(...result.items);
      }

      if (!result || !result.items || result.items.length < limit) {
        hasMore = false;
      } else {
        page++;
      }
    }

    console.log(`✅ Total ${allAssets.length} aset ditemukan. Mem-filter...`);

    // Normalize using helper and only keep fungible token shapes
    const formattedAssets = allAssets
      .map(a => normalizeHeliusAsset(a, owner))
      .filter(x => x && Number(x.total) > 0);

    // Convert to aggregated tokens by mint (module's expected format)
    const map = new Map();
    for (const a of formattedAssets) {
      const mint = a.mint;
      if (!mint) continue;
      const cur = map.get(mint) || { mint, total: 0, decimals: a.decimals || 0, accounts: [], token_name: a.name || null, symbol: a.symbol || null };
      cur.total = (Number(cur.total) || 0) + (Number(a.total) || 0);
      // merge accounts
      if (Array.isArray(a.accounts)) {
        for (const acc of a.accounts) cur.accounts.push(acc);
      } else {
        cur.accounts.push({ address: owner, amount: a.total });
      }
      if (!cur.symbol && a.symbol) cur.symbol = a.symbol;
      if (!cur.token_name && a.name) cur.token_name = a.name;
      map.set(mint, cur);
    }

    const tokens = Array.from(map.values()).sort((x, y) => y.total - x.total);
    const resultObj = { owner, tokens, totalTokenAccounts: formattedAssets.length };
    CACHE.walletAssets.set(owner, { value: resultObj, expires: Date.now() + WALLET_TTL_MS });

    console.timeEnd(`getWalletAssets:${owner}`);
    return resultObj;
  } catch (err) {
    console.error(`❌ Gagal mengambil aset untuk ${owner}:`, err && err.message ? err.message : err);
    console.timeEnd(`getWalletAssets:${owner}`);
    return { owner, tokens: [], totalTokenAccounts: 0, error: err && err.message ? err.message : String(err) };
  }
}

/**
 * Get wallet assets plus USD valuation per token and total USD sum.
 * Returns: { owner, tokens: [{ mint, total, decimals, usdPrice, usdValue, accounts }], totalUsd }
 */
async function getWalletTotalAssets(owner, opts = {}) {
  try {
    const base = await getAssetsByOwner(owner, opts);
    if (!base || !Array.isArray(base.tokens)) return { owner, tokens: [], totalUsd: 0 };

    // fetch prices in parallel (best-effort)
    const tokensWithPrices = await Promise.all(base.tokens.map(async t => {
      let price = null;
      try {
        price = await fetchTokenPriceUsd(t.mint);
      } catch (e) {
        price = null;
      }
      const usdValue = (price !== null && !isNaN(price)) ? (Number(t.total || 0) * Number(price)) : null;
      return Object.assign({}, t, { usdPrice: price, usdValue });
    }));

    const totalUsd = tokensWithPrices.reduce((s, x) => s + (Number(x.usdValue || 0) || 0), 0);
    return { owner, tokens: tokensWithPrices, totalUsd };
  } catch (err) {
    return { owner, tokens: [], totalUsd: 0, error: err.message || String(err) };
  }
}

// export new functions (Helius-only asset lookup)
module.exports = Object.assign(module.exports || {}, { getAssetsByOwner, getWalletTotalAssets });
  
