require('dotenv').config();
const { Connection, PublicKey } = require('@solana/web3.js');
const axios = require('axios');
const moment = require('moment');
const { getSolSnifferData, analyzeSolSnifferSecurity } = require('../utils/solSnifferClient');

const SOLANA_RPC = process.env.SOLANA_RPC || 'https://api.mainnet-beta.solana.com';
const connection = new Connection(SOLANA_RPC);

/**
 * Get Solana token details
 */
async function getSolanaTokenDetails(address) {
  try {
    console.log(`Getting Solana token details for ${address}`);
    const tokenPublicKey = new PublicKey(address);
    
    // Get token account info
    const tokenInfo = await connection.getParsedAccountInfo(tokenPublicKey);
    
    if (!tokenInfo.value || !tokenInfo.value.data) {
      throw new Error('Token not found');
    }
    
    // Fetch SolSniffer data for enhanced security analysis
    const solSnifferData = await getSolSnifferData(address);
    console.log('SolSniffer data received:', solSnifferData ? 'Yes' : 'No');
    
    let name, symbol, decimals, totalSupply, owner;
    
    // Use SolSniffer data if available, otherwise fallback to traditional methods
    if (solSnifferData && solSnifferData.tokenData) {
      const { tokenData, tokenInfo: tokenInfoSniffer } = solSnifferData;
      
      name = tokenData.tokenName || await getTokenName(tokenPublicKey);
      symbol = tokenData.tokenSymbol || await getTokenSymbol(tokenPublicKey, address);
      decimals = 9; // Default for Solana tokens
      totalSupply = tokenInfoSniffer?.supplyAmount || await getTokenSupply(tokenPublicKey);
      owner = tokenData.tokenOverview?.deployer || await getTokenOwner(tokenPublicKey);
      
      // Security analysis using SolSniffer data
      const securityAnalysis = analyzeSolSnifferSecurity(solSnifferData);
      
      // Log security findings
      console.log(`Solana token security analysis: ${securityAnalysis.securityStatus}`);
      if (securityAnalysis.warnings.length > 0) {
        console.log('Security warnings:', securityAnalysis.warnings.join(', '));
      }
      
      // Parse externals for social links
      const socials = { website: null, twitter: null, telegram: null };
      if (tokenData.externals) {
        try {
          const externals = typeof tokenData.externals === 'string' ? 
            JSON.parse(tokenData.externals) : tokenData.externals;
            
          if (externals.website) {
            socials.website = externals.website;
          }
          if (externals.twitter_handle) {
            socials.twitter = `https://twitter.com/${externals.twitter_handle}`;
          }
        } catch (e) {
          console.error('Error parsing externals:', e);
        }
      }
      
      // Create creation info from deploy time
      const creationInfo = {
        timestamp: tokenData.deployTime ? 
          moment(tokenData.deployTime).unix() : 
          Math.floor(Date.now() / 1000) - 86400
      };
      
      // Get token age
      const age = moment().diff(moment.unix(creationInfo.timestamp), 'seconds');
      const ageFormatted = formatAge(age);
      
      // Process holder information
      let holders = { count: 0, top: [] };
      if (tokenData.ownersList && tokenData.ownersList.length > 0) {
        holders.count = tokenData.ownersList.length;
        holders.top = tokenData.ownersList.map(h => ({
          address: h.address,
          balance: h.amount,
          percentage: parseFloat(h.percentage)
        }));
      }
      
      // Process liquidity information
      let lpInfo = {
        dex: 'Unknown',
        lpAddress: 'Unknown',
        liquidity: 'Unknown',
        locked: solSnifferData.tokenData.auditRisk?.lpBurned ? 'LOCKED' : 'Unknown'
      };
      
      if (tokenData.liquidityList && tokenData.liquidityList.length > 0) {
        const dexEntries = Object.entries(tokenData.liquidityList[0]);
        if (dexEntries.length > 0) {
          const [dexName, dexInfo] = dexEntries[0];
          lpInfo.dex = dexName;
          lpInfo.liquidity = dexInfo.amount ? `$${dexInfo.amount.toLocaleString()}` : 'Unknown';
          lpInfo.lpAddress = dexInfo.address || 'Unknown';
        }
      }
      
      // Tax information (Solana typically doesn't have buy/sell taxes)
      const taxInfo = await getTaxInfo(tokenPublicKey);
      
      // Add additional data from SolSniffer for security analysis
      return {
        name,
        symbol,
        decimals,
        totalSupply,
        owner,
        contractAddress: address,
        creationDate: creationInfo.timestamp,
        ageFormatted,
        holders,
        lpInfo,
        taxInfo,
        socials,
        verified: true, // All Solana tokens with SolSniffer data are verified
        solSnifferScore: solSnifferData.tokenData.score,
        // Add security-specific flags for analysis
        mintable: solSnifferData.tokenData.auditRisk?.mintDisabled === false,
        freezable: solSnifferData.tokenData.auditRisk?.freezeDisabled === false,
        securityWarnings: securityAnalysis.warnings
      };
    } else {
      // When SolSniffer data isn't available, try to fetch real data from DexScreener
      // and the Solana blockchain instead of using placeholders
      try {
        // Try to get data directly from DexScreener first (most reliable for public tokens)
        const dexScreenerData = await fetchBasicTokenInfoFromDexScreener(address);
        if (dexScreenerData) {
          name = dexScreenerData.name || await getTokenName(tokenPublicKey);
          symbol = dexScreenerData.symbol || await getTokenSymbol(tokenPublicKey, address);
          
          console.log(`Found token data from DexScreener: ${symbol}`);
        } else {
          // If DexScreener doesn't have data, try to get from the blockchain
          name = await getTokenName(tokenPublicKey);
          symbol = await getTokenSymbol(tokenPublicKey, address);
        }
      } catch (error) {
        console.error('Error getting token metadata:', error);
        name = await getTokenName(tokenPublicKey);
        symbol = await getTokenSymbol(tokenPublicKey, address);
      }
      
      decimals = await getTokenDecimals(tokenPublicKey);
      totalSupply = await getTokenSupply(tokenPublicKey);
      owner = await getTokenOwner(tokenPublicKey);
      
      // Get token creation date - simplified
      const creationInfo = await getTokenCreationInfo(tokenPublicKey);
      
      // Get top holders
      const holders = await getTopHolders(tokenPublicKey);
      
      // Check LP info
      const lpInfo = await getLPInfo(tokenPublicKey);
      
      // Get token age
      const age = moment().diff(moment.unix(creationInfo.timestamp), 'seconds');
      const ageFormatted = formatAge(age);
      
      // Get tax information (if applicable)
      const taxInfo = await getTaxInfo(tokenPublicKey);
      
      return {
        name,
        symbol,
        decimals,
        totalSupply,
        owner,
        contractAddress: address,
        creationDate: creationInfo.timestamp,
        ageFormatted,
        holders,
        lpInfo,
        taxInfo
      };
    }
  } catch (error) {
    console.error('Error getting Solana token details:', error);
    throw new Error('Failed to get Solana token details: ' + error.message);
  }
}

/**
 * Attempt to fetch basic token info from DexScreener
 */
async function fetchBasicTokenInfoFromDexScreener(address) {
  try {
    const apiUrl = `https://api.dexscreener.com/latest/dex/tokens/${address}`;
    const response = await axios.get(apiUrl, {
      headers: {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) Chrome/120.0.0.0'
      },
      timeout: 10000
    });
    
    if (response.data && response.data.pairs && response.data.pairs.length > 0) {
      // Get the first pair with data
      const firstPair = response.data.pairs[0];
      if (firstPair.baseToken) {
        return {
          name: firstPair.baseToken.name,
          symbol: firstPair.baseToken.symbol
        };
      }
    }
    return null;
  } catch (error) {
    console.error('Error fetching from DexScreener:', error.message);
    return null;
  }
}

/**
 * Get token name
 */
async function getTokenName(tokenPublicKey) {
  try {
    // This is a simplified implementation
    // In a real scenario, you'd need to fetch from token metadata program
    return "SOL Token"; // Placeholder
  } catch (error) {
    return "Unknown";
  }
}

/**
 * Get token symbol - improved to attempt getting real symbol
 */
async function getTokenSymbol(tokenPublicKey, address) {
  try {
    // Try to get token metadata from Solana blockchain
    try {
      const info = await connection.getAccountInfo(tokenPublicKey);
      if (info && info.data) {
        // For SPL tokens, try to parse the symbol
        // Note: This is a simplified approach
        const dataStr = info.data.toString();
        if (dataStr.length > 50) {
          // Look for common symbol patterns - this is a heuristic approach
          const match = dataStr.match(/[A-Z0-9]{2,10}/);
          if (match && match[0]) {
            return match[0];
          }
        }
      }
    } catch (error) {
      console.log('Error parsing token symbol:', error.message);
    }
    
    // If we couldn't get the symbol, use a shortened version of the address
    // instead of a generic placeholder
    if (address) {
      return address.substring(0, 4) + "..." + address.substring(address.length - 4);
    }
    
    return "UNKNOWN"; // Last resort fallback
  } catch (error) {
    return "UNKNOWN";
  }
}

/**
 * Get token decimals
 */
async function getTokenDecimals(tokenPublicKey) {
  try {
    // This is a simplified implementation
    return 9; // Most Solana tokens use 9 decimals
  } catch (error) {
    return 9;
  }
}

/**
 * Get token supply
 */
async function getTokenSupply(tokenPublicKey) {
  try {
    const supply = await connection.getTokenSupply(tokenPublicKey);
    return supply.value.uiAmount;
  } catch (error) {
    return 0;
  }
}

/**
 * Get token owner/authority
 */
async function getTokenOwner(tokenPublicKey) {
  try {
    console.log('Getting token owner/authority for Solana token...');
    
    // Method 1: Get mint authority from the token's mint account
    try {
      const mintInfo = await connection.getAccountInfo(tokenPublicKey);
      
      if (mintInfo && mintInfo.data) {
        // Solana token mint accounts have the mint authority at a specific offset
        // This is a simplified approach - proper parsing would depend on the SPL token program layout
        const data = mintInfo.data;
        
        // Extract mint authority if present
        // Note: This is an approximation and may need adjustment based on the SPL token program layout
        if (data.length >= 82) {
          // Check if authority is enabled (not null)
          const authorityOptionByte = data[33];  // Position may vary
          
          if (authorityOptionByte === 1) {
            // Extract pubkey bytes - position may vary
            const authorityBytes = data.slice(34, 66);
            const authorityAddress = new PublicKey(authorityBytes).toBase58();
            
            // Check if it's the null address
            if (authorityAddress === '11111111111111111111111111111111') {
              console.log('Mint authority is null address (renounced)');
              return 'RENOUNCED';
            }
            
            console.log(`Found mint authority: ${authorityAddress}`);
            return authorityAddress;
          } else {
            console.log('No mint authority found - might be renounced');
            return 'RENOUNCED';
          }
        }
      }
    } catch (error) {
      console.error('Error getting mint authority:', error.message);
    }
    
    // Method 2: Try to get the token creator from transaction history
    try {
      console.log('Attempting to find token creator from transaction history...');
      
      // Get the 10 earliest transactions for this address
      const signatures = await connection.getSignaturesForAddress(tokenPublicKey, { limit: 10, before: '' });
      
      if (signatures && signatures.length > 0) {
        // Sort by oldest first to find the creation transaction
        const sortedSignatures = signatures.sort((a, b) => {
          const timeA = a.blockTime || 0;
          const timeB = b.blockTime || 0;
          return timeA - timeB;
        });
        
        // Get the earliest transaction
        const earliestTx = await connection.getTransaction(sortedSignatures[0].signature);
        
        if (earliestTx && earliestTx.transaction && earliestTx.transaction.signatures) {
          // The first signer is typically the creator
          const creatorSignature = earliestTx.transaction.signatures[0];
          const creatorAddress = earliestTx.transaction.message.accountKeys[0].toBase58();
          
          console.log(`Found likely creator: ${creatorAddress}`);
          return creatorAddress;
        }
      }
    } catch (error) {
      console.error('Error getting token creator from transactions:', error.message);
    }
    
    // Method 3: Try to get creator from token metadata program
    try {
      console.log('Attempting to find creator from token metadata program...');
      
      // Placeholder: This would involve fetching token metadata from the Metaplex program
      // This is more complex and would require additional dependencies and implementation
    } catch (error) {
      console.error('Error getting creator from metadata program:', error.message);
    }
    
    return "Unknown"; // Default if all methods fail
  } catch (error) {
    console.error('Error in getTokenOwner:', error.message);
    return "Unknown";
  }
}

/**
 * Get token creation information
 */
async function getTokenCreationInfo(tokenPublicKey) {
  try {
    // This is a simplified implementation
    // In a real scenario, you'd need to find the transaction that created the token
    return {
      creator: "Unknown",
      timestamp: Math.floor(Date.now() / 1000) - 86400 // Default to 1 day ago
    };
  } catch (error) {
    return {
      timestamp: Math.floor(Date.now() / 1000) - 86400
    };
  }
}

/**
 * Get top token holders
 */
async function getTopHolders(tokenPublicKey) {
  try {
    // This is a simplified implementation
    // In a real scenario, you'd query token accounts
    return {
      count: 100, // Placeholder
      top: []
    };
  } catch (error) {
    return { count: 0, top: [] };
  }
}

/**
 * Get LP information - typically on Raydium or other Solana DEXes
 */
async function getLPInfo(tokenPublicKey) {
  // Simplified implementation
  return {
    dex: 'Raydium',
    lpAddress: 'Unknown',
    liquidity: 'Unknown',
    locked: 'Unknown'
  };
}

/**
 * Get tax information
 */
async function getTaxInfo(tokenPublicKey) {
  // Simplified implementation
  return {
    buy: '0.0%',
    sell: '0.0%'
  };
}

/**
 * Format age to human-readable string
 */
function formatAge(seconds) {
  const days = Math.floor(seconds / 86400);
  const hours = Math.floor((seconds % 86400) / 3600);
  const minutes = Math.floor((seconds % 3600) / 60);
  
  return `${days}d - ${hours}h ${minutes}m ${seconds % 60}s`;
}

module.exports = {
  getSolanaTokenDetails
};
