require('dotenv').config();
const TronWeb = require('@tronweb3/tronweb');
const axios = require('axios');
const moment = require('moment');

const TRON_RPC = process.env.TRON_RPC || 'https://api.trongrid.io';
const tronWeb = new TronWeb({
  fullHost: TRON_RPC
});

/**
 * Get TRON token details
 */
async function getTronTokenDetails(address) {
  try {
    // Convert address format if needed
    const contractAddress = address;
    
    // Get basic token info
    const contract = await tronWeb.contract().at(contractAddress);
    
    const name = await contract.name().call();
    const symbol = await contract.symbol().call();
    const decimals = await contract.decimals().call();
    const totalSupply = await contract.totalSupply().call();
    
    // Convert totalSupply to human-readable format
    const totalSupplyFormatted = totalSupply / Math.pow(10, decimals);
    
    // Get owner address (may fail if not implemented)
    let owner = 'Unknown';
    try {
      owner = await contract.owner().call();
    } catch (error) {
      console.log('Owner method not available');
    }
    
    // Get token contract creation date
    const creationInfo = await getContractCreationInfo(contractAddress);
    
    // Get top holders
    const holders = await getTopHolders(contractAddress);
    
    // Check LP info
    const lpInfo = await getLPInfo(contractAddress);
    
    // Get token age
    const age = moment().diff(moment.unix(creationInfo.timestamp), 'seconds');
    const ageFormatted = formatAge(age);
    
    // Get tax information (buy/sell)
    const taxInfo = await getTaxInfo(contractAddress);
    
    return {
      name,
      symbol,
      decimals,
      totalSupply: totalSupplyFormatted,
      owner,
      contractAddress,
      creationDate: creationInfo.timestamp,
      ageFormatted,
      holders,
      lpInfo,
      taxInfo
    };
  } catch (error) {
    console.error('Error getting TRON token details:', error);
    throw new Error('Failed to get TRON token details');
  }
}

/**
 * Get contract creation information
 */
async function getContractCreationInfo(address) {
  try {
    // This is a simplified implementation
    // In a real scenario, you'd query the TRON API for contract creation
    return {
      creator: "Unknown",
      timestamp: Math.floor(Date.now() / 1000) - 86400 // Default to 1 day ago
    };
  } catch (error) {
    return { timestamp: Math.floor(Date.now() / 1000) - 86400 };
  }
}

/**
 * Get top token holders
 */
async function getTopHolders(address) {
  try {
    // This is a simplified implementation
    // In a real scenario, you'd query TRON API for token holders
    return {
      count: 100, // Placeholder
      top: []
    };
  } catch (error) {
    return { count: 0, top: [] };
  }
}

/**
 * Get LP information - typically on JustSwap/SunSwap
 */
async function getLPInfo(address) {
  // Simplified implementation
  return {
    dex: 'SunSwap',
    lpAddress: 'Unknown',
    liquidity: 'Unknown',
    locked: 'Unknown'
  };
}

/**
 * Get tax information
 */
async function getTaxInfo(address) {
  // Simplified implementation
  return {
    buy: '0.0%',
    sell: '0.0%'
  };
}

/**
 * Format age to human-readable string
 */
function formatAge(seconds) {
  const days = Math.floor(seconds / 86400);
  const hours = Math.floor((seconds % 86400) / 3600);
  const minutes = Math.floor((seconds % 3600) / 60);
  
  return `${days}d - ${hours}h ${minutes}m ${seconds % 60}s`;
}

module.exports = {
  getTronTokenDetails
};
