const axios = require("axios");

// 🔹 Ganti ini dengan API Key kamu dari https://www.helius.dev
const API_KEY = "207115c7-67ef-48e8-860c-ae20fbfe1b37";

// 🔹 Endpoint Helius RPC
const RPC_URL = `https://mainnet.helius-rpc.com/?api-key=${API_KEY}`;

// 🔹 Ganti ini dengan mint address token yang ingin kamu analisa
const MINT_ADDRESS = "2iKoxXwc9GPKUQWth2JcdWJXc7rV5Sp4ErBjQSr2pump"; // contoh: USDC

// 🔁 Fungsi helper dengan retry otomatis (tahan rate limit)
async function safePost(data, retries = 3) {
  for (let i = 0; i < retries; i++) {
    try {
      return await axios.post(RPC_URL, data);
    } catch (err) {
      if (err.response?.status === 429) {
        console.log(`⚠️ Rate limited, retrying (${i + 1}/${retries})...`);
        // Gunakan backoff eksponensial sederhana
        await new Promise(r => setTimeout(r, 1000 * Math.pow(2, i)));
      } else {
        // Tampilkan error yang lebih detail
        console.error("Axios Error:", err.response?.data || err.message);
        throw err;
      }
    }
  }
  throw new Error("Failed after retries");
}

async function getTopHolders(mint, topN = 10) {
  try {
    console.time("getTopHolders");
    console.log(`🔍 Menganalisa top ${topN} holder untuk mint: ${mint}`);

    const TOKEN_PROGRAM_ID = "TokenkegQfeZyiNwAJbNbGKPFXCWuBvf9Ss623VQ5DA";
    const MAX_ACCOUNTS = 20000; // batas aman default

    let allAccounts = null;
    try {
      console.log("🚀 Mencoba Strategi 1: getProgramAccounts (Akurat)...");
      const progRes = await safePost({
        jsonrpc: "2.0",
        id: 1,
        method: "getProgramAccounts",
        params: [
          TOKEN_PROGRAM_ID,
          {
            encoding: "jsonParsed",
            filters: [
              // 💡 PERBAIKAN 1: Tambahkan dataSize filter
              // Ini sangat penting untuk memfilter HANYA token account
              { dataSize: 165 },
              // Filter berdasarkan mint address (offset 0)
              { memcmp: { offset: 0, bytes: mint } },
            ],
          },
        ],
      });

      allAccounts = progRes.data.result;
      if (!Array.isArray(allAccounts)) allAccounts = [];
      console.log(`✅ getProgramAccounts berhasil: ${allAccounts.length} token accounts ditemukan.`);

      if (allAccounts.length > MAX_ACCOUNTS) {
        console.warn(`⚠️ getProgramAccounts mengembalikan ${allAccounts.length} akun, membatasi ke ${MAX_ACCOUNTS} pertama untuk performa.`);
        allAccounts = allAccounts.slice(0, MAX_ACCOUNTS);
      }
    } catch (e) {
      // Jika getProgramAccounts gagal, fallback
      console.warn(`⚠️ Strategi 1 gagal — fallback ke Strategi 2: getTokenLargestAccounts...`, e.message);
      const res = await safePost({
        jsonrpc: "2.0",
        id: 1,
        method: "getTokenLargestAccounts",
        params: [mint],
      });

      if (!res?.data?.result?.value) {
        throw new Error('Invalid response from getTokenLargestAccounts');
      }

      // Ambil top 20 (default dari getTokenLargestAccounts)
      const sample = res.data.result.value;
      console.log(`✅ getTokenLargestAccounts berhasil: ${sample.length} akun terbesar ditemukan.`);

      // batch getMultipleAccounts untuk semua sample addresses
      const addresses = sample.map(a => a.address);
      let accountsInfo = null;
      try {
        console.log(`batch-fetching ${addresses.length} account details...`);
        const batchRes = await safePost({
          jsonrpc: "2.0",
          id: 1,
          method: "getMultipleAccounts",
          params: [addresses, { encoding: "jsonParsed" }],
        });
        accountsInfo = batchRes.data.result.value;
      } catch (err2) {
        // Fallback jika getMultipleAccounts gagal (jarang terjadi)
        console.warn("⚠️ Batch getMultipleAccounts gagal, mencoba per-account:", err2.message);
        accountsInfo = [];
        for (const addr of addresses) {
          const accountRes = await safePost({
            jsonrpc: "2.0",
            id: 1,
            method: "getAccountInfo",
            params: [addr, { encoding: "jsonParsed" }],
          });
          accountsInfo.push(accountRes.data.result?.value ?? null);
          await new Promise(r => setTimeout(r, 100)); // rate limit kecil
        }
      }

      // 💡 PERBAIKAN 2: Ubah cara membangun 'allAccounts'
      // accountsInfo[i] SUDAH merupakan objek akun penuh
      allAccounts = sample.map((s, i) => ({
        pubkey: s.address,
        account: accountsInfo[i], // Langsung tetapkan objek akun
      }));
    }

    // --- Parsing dan Agregasi (Logika ini sudah benar) ---

    console.log("🔄 Mem-parsing dan meng-agregasi data akun...");
    const perAccount = [];
    for (const item of allAccounts) {
      if (!item || !item.account) continue; // Skip jika akun null (mungkin ditutup)

      const parsed = item.account?.data?.parsed?.info;
      if (!parsed) continue; // skip non-parsed entries

      const owner = parsed.owner ?? "Unknown";
      const tokenAmount = parsed.tokenAmount ?? {};
      
      // Ambil uiAmount jika ada, jika tidak hitung manual
      const amount = typeof tokenAmount.uiAmount === 'number'
        ? tokenAmount.uiAmount
        : (Number(tokenAmount.amount ?? 0) / Math.pow(10, tokenAmount.decimals ?? 6));

      if (amount > 0) { // Hanya proses akun dengan saldo
        perAccount.push({ tokenAccount: item.pubkey, owner, amount });
      }
    }

    // AGREGASI per owner
    const ownerMap = new Map();
    for (const a of perAccount) {
      const key = a.owner;
      const cur = ownerMap.get(key) || { owner: key, total: 0, accounts: [] };
      cur.total += a.amount;
      cur.accounts.push({ tokenAccount: a.tokenAccount, amount: a.amount });
      ownerMap.set(key, cur);
    }

    const aggregated = Array.from(ownerMap.values())
      .sort((x, y) => y.total - x.total)
      .slice(0, topN);

    // Hitung persentase berdasarkan total yang diagregasi (bukan hanya topN)
    const totalInAggregated = Array.from(ownerMap.values()).reduce((s, v) => s + v.total, 0);

    const finalResult = aggregated.map((item, i) => ({
      rank: i + 1,
      owner: item.owner,
      total: item.total,
      percent: totalInAggregated ? (item.total / totalInAggregated) * 100 : 0,
      accounts: item.accounts,
    }));


    console.timeEnd("getTopHolders");

    // --- Output ---
    console.log(`\n💎 Top ${finalResult.length} Wallets (by Owner) for ${mint}\n`);
    for (const a of finalResult) {
      const short = `${a.owner.slice(0, 4)}...${a.owner.slice(-4)}`;
      const pct = a.percent.toFixed(2) + '%';
      const emoji = a.rank === 1 ? '🥇' : (a.rank === 2 ? '🥈' : (a.rank === 3 ? '🥉' : '🔸'));
      console.log(`#${a.rank} ${emoji} ${short} - Total: ${a.total.toLocaleString()} (${pct}) - [${a.accounts.length} akun]`);
    }

    console.log('\nTop holders (aggregated by owner):');
    console.table(finalResult.map(a => ({
      rank: a.rank,
      owner: a.owner,
      total: a.total,
      percent: a.percent.toFixed(2) + '%',
      num_accounts: a.accounts.length
    })));

    if (finalResult.length > 0) {
      console.log(`\nDetail accounts for Top Holder #${finalResult[0].rank} (${finalResult[0].owner}):`);
      // 💡 PERBAIKAN 3: Menghapus blok console.table duplikat
      console.table(finalResult[0].accounts.sort((a,b) => b.amount - a.amount));
    }
    
    // Return an object (JSON-like) so callers can parse programmatically
    return { items: finalResult };
  } catch (err) {
    const message = err.response?.data?.error?.message || err.message;
    console.error("❌ Error:", message);
    // Return error in JSON-like structure for programmatic callers
    return { items: [], error: message };
  }
}

// Jalankan (boleh passing mint dan topN via command-line)
if (require.main === module) {
  const argvMint = process.argv[2];
  const argvTop = Number(process.argv[3]) || 10;
  const mintToUse = argvMint || MINT_ADDRESS;

  (async () => {
    const res = await getTopHolders(mintToUse, argvTop);
    // If CLI, pretty print JSON
    console.log('\nJSON output:');
    console.log(JSON.stringify(res, null, 2));
  })().catch(e => console.error('Error running getTopHolders:', e && e.message));
}

// Export the function for other modules to consume (returns JSON-like object)
module.exports = { getTopHolders };