const path = require('path');
const fs = require('fs');

/**
 * Generate DexScreener data display format for tokens
 * @param {Object} dexData - The dex data from DexScreener
 * @returns {Object} Formatted data including text and image path
 */
function generateDexDataDisplay(dexData) {
  if (!dexData || !dexData.extendedDexData) {
    return {
      text: "No DexScreener data available",
      imagePath: null
    };
  }
  
  const extended = dexData.extendedDexData;
  
  // Format price change with color indicator
  const formatPriceChange = (change) => {
    if (change === undefined || change === null) return "N/A";
    const value = parseFloat(change);
    
    if (value > 0) return `+${value.toFixed(2)}%`;
    if (value < 0) return `${value.toFixed(2)}%`;
    return "0.00%";
  };
  
  // Format volume with K, M, B suffixes
  const formatVolume = (vol) => {
    if (vol === undefined || vol === null) return "N/A";
    const value = parseFloat(vol);
    
    if (value >= 1e9) return `$${(value / 1e9).toFixed(2)}B`;
    if (value >= 1e6) return `$${(value / 1e6).toFixed(2)}M`;
    if (value >= 1e3) return `$${(value / 1e3).toFixed(2)}K`;
    return `$${value.toFixed(2)}`;
  };
  
  // Format transactions count
  const formatTxns = (txns) => {
    if (!txns) return "N/A";
    return `${txns.buys || 0} buys, ${txns.sells || 0} sells`;
  };
  
  // Create detailed text display
  let text = `📊 <b>DexScreener Data: ${extended.symbol}</b>\n\n`;
  
  // Price section
  text += `<b>Price:</b> $${parseFloat(extended.price).toFixed(8)}\n`;
  text += `<b>Price Change:</b>\n`;
  text += `  • 5m: ${formatPriceChange(extended.priceChange.m5)}\n`;
  text += `  • 1h: ${formatPriceChange(extended.priceChange.h1)}\n`;
  text += `  • 6h: ${formatPriceChange(extended.priceChange.h6)}\n`;
  text += `  • 24h: ${formatPriceChange(extended.priceChange.h24)}\n\n`;
  
  // Volume section
  text += `<b>Volume:</b>\n`;
  text += `  • 5m: ${formatVolume(extended.volume.m5)}\n`;
  text += `  • 1h: ${formatVolume(extended.volume.h1)}\n`;
  text += `  • 6h: ${formatVolume(extended.volume.h6)}\n`;
  text += `  • 24h: ${formatVolume(extended.volume.h24)}\n\n`;
  
  // Transactions section if available
  if (extended.txns) {
    text += `<b>Transactions:</b>\n`;
    text += `  • 1h: ${formatTxns(extended.txns.h1)}\n`;
    text += `  • 6h: ${formatTxns(extended.txns.h6)}\n`;
    text += `  • 24h: ${formatTxns(extended.txns.h24)}\n\n`;
  }
  
  // Pair information
  text += `<b>Pair Info:</b>\n`;
  text += `  • DEX: ${extended.dexId}\n`;
  text += `  • Pair: ${extended.pairAddress.substring(0, 8)}...${extended.pairAddress.substring(extended.pairAddress.length - 8)}\n`;
  if (extended.pairCreatedAt) {
    text += `  • Created: ${extended.pairCreatedAt.toISOString().split('T')[0]}\n`;
  }
  
  // Add DexScreener link
  text += `\n<a href="${extended.url}">🔍 View on DexScreener</a>\n`;
  
  // Get the DexScreener image path
  const imagePath = path.join(__dirname, '..', 'assets', 'dex.png');
  
  // Check if image exists
  const imageExists = fs.existsSync(imagePath);
  
  return {
    text: text,
    imagePath: imageExists ? imagePath : null
  };
}

module.exports = {
  generateDexDataDisplay
};
