const { createCanvas, loadImage, registerFont } = require('canvas');
const path = require('path');
const fs = require('fs');
const { parseMarketCap } = require('./numberUtils');

/**
 * Generate an image with token analysis results
 * @param {Object} tokenDetails - Token information
 * @param {Object} dexData - Data from DEX
 * @param {Object} securityAnalysis - Security analysis results
 * @param {String} network - Blockchain network
 */
async function generateResultImage(tokenDetails, dexData, securityAnalysis, network) {
  try {
    // Define canvas dimensions
    const width = 1200;
    const height = 675;
    
    // Create canvas
    const canvas = createCanvas(width, height);
    const ctx = canvas.getContext('2d');
    
    // Load template image
    const templatePath = path.join(__dirname, '..', 'assets', 'result.png');
    if (!fs.existsSync(templatePath)) {
      console.error('Template image not found at:', templatePath);
      return null;
    }
    
    // Load token logo from DexScreener if available
    let tokenLogoImage = null;
    
    try {
      // Try to get the token logo URL from DexScreener data
      let tokenLogoUrl = null;
      
      if (dexData && dexData.rawData && dexData.rawData.info && dexData.rawData.info.imageUrl) {
        tokenLogoUrl = dexData.rawData.info.imageUrl;
        console.log(`Found token logo URL in DexScreener data: ${tokenLogoUrl}`);
        
        // Add proxy prefix for CORS issues and add error handling
        try {
          // Add a timeout to prevent hanging on bad URLs
          const imageLoadPromise = new Promise((resolve, reject) => {
            // Attempt to load the image
            const img = new Image();
            img.onload = () => resolve(img);
            img.onerror = () => reject(new Error("Failed to load image"));
            
            // Set a timeout
            const timeout = setTimeout(() => {
              reject(new Error("Image load timeout"));
            }, 5000);
            
            // Try to load the image
            img.src = tokenLogoUrl;
            
            // Clear the timeout if the image loads or errors out
            img.onload = () => {
              clearTimeout(timeout);
              resolve(img);
            };
            
            img.onerror = () => {
              clearTimeout(timeout);
              reject(new Error("Failed to load image"));
            };
          });
          
          tokenLogoImage = await imageLoadPromise;
          console.log('Successfully loaded token logo image');
        } catch (err) {
          console.log(`Error loading image from URL, trying fallback: ${err.message}`);
          tokenLogoImage = null;
        }
      } else {
        console.log('No token logo URL found in DexScreener data');
      }
    } catch (logoError) {
      console.error('Error loading token logo:', logoError);
    }
    
    // Load and draw the background template
    const backgroundImage = await loadImage(templatePath);
    ctx.drawImage(backgroundImage, 0, 0, width, height);
    
    // Draw token logo if available
    if (tokenLogoImage) {
      // Calculate position to place it on the middle-left side
      const logoSize = 180;  // Size for both width and height to keep it square
      const logoX = 80;      // Left position
      const logoY = height / 2 - logoSize / 2;  // Vertically centered
      
      // Draw the logo with a circular mask
      ctx.save();
      ctx.beginPath();
      ctx.arc(logoX + logoSize / 2, logoY + logoSize / 2, logoSize / 2, 0, Math.PI * 2, true);
      ctx.closePath();
      ctx.clip();
      
      // Draw the logo image inside the circular clip
      ctx.drawImage(tokenLogoImage, logoX, logoY, logoSize, logoSize);
      ctx.restore();
      
      console.log('Drew token logo on image');
    } else {
      console.log('Token logo not available, skipping logo display');
    }
    
    // Set font and colors
    ctx.font = 'bold 48px Arial';
    ctx.fillStyle = '#FFFFFF';
    ctx.textAlign = 'center';
    
    // Draw token symbol and security status
    ctx.fillText(`${tokenDetails.symbol || "Unknown"} Analysis`, width / 2, 80);
    
    // Set security status color based on the analysis
    let statusColor = '#FFFFFF';
    switch(securityAnalysis.status) {
      case 'SAFE': 
        statusColor = '#4CAF50'; break;
      case 'LIKELY SAFE': 
        statusColor = '#8BC34A'; break;
      case 'CAUTION': 
        statusColor = '#FFC107'; break;
      case 'HIGH RISK': 
        statusColor = '#F44336'; break;
      default: 
        statusColor = '#9E9E9E';
    }
    
    // Draw security status
    ctx.fillStyle = statusColor;
    ctx.font = 'bold 40px Arial';
    ctx.fillText(`${securityAnalysis.status}`, width / 2, 140);
    
    // Reset fill style for other text
    ctx.fillStyle = '#FFFFFF';
    ctx.font = '30px Arial';
    ctx.textAlign = 'left';
    
    // Draw token details
    let y = 200;
    const lineHeight = 38;
    
    // Draw network and contract address
    ctx.fillText(`Network: ${network}`, 350, y);
    y += lineHeight;
    
    // Draw contract address (shortened)
    const shortAddr = `${tokenDetails.contractAddress.substring(0, 20)}...`;
    ctx.fillText(`Contract: ${shortAddr}`, 350, y);
    y += lineHeight + 10;
    
    // Draw price and market cap if available
    if (dexData) {
      // Format and display market cap
      // Normalize market cap using shared parser to avoid NaN when dexData.marketCap is a string like '23.4M'
      const mcapNum = parseMarketCap(dexData.marketCap !== undefined ? dexData.marketCap : dexData.mcap);
      let marketCapDisplay = 'Unknown';
      if (mcapNum !== null && !isNaN(mcapNum)) {
        marketCapDisplay = `$${formatNumberWithSuffix(Number(mcapNum))}`;
      } else if (typeof dexData.marketCap === 'string') {
        marketCapDisplay = dexData.marketCap;
      }
      
      ctx.fillText(`Market Cap: ${marketCapDisplay}`, 350, y);
      y += lineHeight;
      
      // Format and display price
      let priceDisplay = "Unknown";
      if (dexData.price) {
        const priceNum = parseFloat(dexData.price);
        const decimals = priceNum < 0.00001 ? 10 : 
                        priceNum < 0.01 ? 6 : 
                        priceNum < 1 ? 4 : 2;
        priceDisplay = `$${priceNum.toFixed(decimals)}`;
      }
      
      ctx.fillText(`Price: ${priceDisplay}`, 350, y);
      y += lineHeight;
      
      // Draw liquidity
      const liquidityText = dexData.liquidityUSD ? 
        `$${formatNumberWithDots(Math.round(dexData.liquidityUSD || 0))}` :
        'Unknown';
      
      ctx.fillText(`Liquidity: ${liquidityText}`, 350, y);
      y += lineHeight;
      
      // Display FDV if available
      if (dexData.fdv) {
        const fdvFormatted = typeof dexData.fdv === 'string' ? dexData.fdv : 
          `$${formatNumberWithSuffix(Number(dexData.fdv))}`;
        ctx.fillText(`FDV: ${fdvFormatted}`, 350, y);
        y += lineHeight;
      }
      
      // Display Volume
      const volumeDisplay = dexData.volume24h ? 
        `$${formatNumberWithDots(Math.round(dexData.volume24h || 0))}` : 'Unknown';
      ctx.fillText(`Volume (24h): ${volumeDisplay}`, 350, y);
      y += lineHeight;
    }
    
    // Draw owner status
    const ownerText = tokenDetails.owner === 'RENOUNCED' ? 
      'Owner: RENOUNCED ✅' : 
      'Owner: NOT RENOUNCED ❌';
    
    ctx.fillText(ownerText, 350, y);
    y += lineHeight;
    
    // Draw LP Lock status
    const lpLockText = tokenDetails.lpInfo && tokenDetails.lpInfo.locked && 
      tokenDetails.lpInfo.locked.includes('LOCKED') ? 
      'LP Locked: YES ✅' : 
      'LP Locked: NO/UNKNOWN ❌';
    
    ctx.fillText(lpLockText, 350, y);
    y += lineHeight;
    
    // Add DEX Paid status if available
    if (dexData && dexData.isPaid !== undefined) {
      const dexPaidText = `DEX Paid: ${dexData.isPaid ? 'YES ✅' : 'NO ❌'}`;
      ctx.fillText(dexPaidText, 350, y);
      y += lineHeight;
    }
    
    // Add honeypot status if available
    if (securityAnalysis.honeypot) {
      const honeypotStatus = securityAnalysis.honeypot;
      let honeypotEmoji = '❓';
      let honeypotDisplay = 'UNKNOWN';
      
      if (honeypotStatus.isHoneypot) {
        honeypotEmoji = '⚠️';
        honeypotDisplay = 'YES';
      } else if (honeypotStatus.confidence === 'Safe' || honeypotStatus.confidence === 'Low Risk') {
        honeypotEmoji = '✅';
        honeypotDisplay = 'NO';
      }
      
      ctx.fillText(`Honeypot: ${honeypotEmoji} ${honeypotDisplay}`, 350, y);
      y += lineHeight;
    }
    
    // Draw token age
    if (tokenDetails.ageFormatted) {
      ctx.fillText(`Age: ${tokenDetails.ageFormatted}`, 350, y);
      y += lineHeight;
    }
    
    // Draw Deepdex branding at the bottom
    ctx.font = 'bold 28px Arial';
    ctx.fillStyle = '#FFFFFF';
    ctx.textAlign = 'center';
  ctx.fillText('Powered by RadarAnalyzer.io', width / 2, height - 40);
    
    // Save image to buffer
    const buffer = canvas.toBuffer('image/png');
    return buffer;
  } catch (error) {
    console.error('Error generating result image:', error);
    return null;
  }
}

// Format number with K, M, B suffix for readability
function formatNumberWithSuffix(num) {
  if (num >= 1e9) return (num / 1e9).toFixed(1) + 'B';
  if (num >= 1e6) return (num / 1e6).toFixed(1) + 'M';
  if (num >= 1e3) return (num / 1e3).toFixed(1) + 'K';
  return num.toFixed(2);
}

// Format number with dots as thousands separators
function formatNumberWithDots(num) {
  return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ".");
}

module.exports = {
  generateResultImage
};

// Generate a PnL card showing change between first call Mcap and latest Mcap
async function generatePnlCard(opts) {
  const { symbol = '', firstMcapRaw, latestMcapRaw, firstCalledBy = '', firstCalledAt = null, latestCalledAt = null } = opts || {};
  try {
    const width = 1000;
    const height = 540;
    const canvas = createCanvas(width, height);
    const ctx = canvas.getContext('2d');

    // Load base background
    const bgPath = path.join(__dirname, '..', 'assets', 'pnlcalled.png');
    if (!fs.existsSync(bgPath)) {
      console.error('PNL background not found at:', bgPath);
      return null;
    }
    const bg = await loadImage(bgPath);
    ctx.drawImage(bg, 0, 0, width, height);

    // Parse mcaps
    const firstNum = parseMarketCap(firstMcapRaw);
    const latestNum = parseMarketCap(latestMcapRaw);

    // Compute percent change safely
    let pct = null;
    if (firstNum !== null && latestNum !== null && firstNum > 0) {
      pct = ((latestNum - firstNum) / firstNum) * 100;
    }

    // Allow optional top margin to avoid overlapping artwork in the template
  // marginTop controls vertical offset to avoid overlapping artwork. Add a small
  // extraOffset (5-10px) for fine-tuning so text doesn't hug the top edge.
  const marginTop = (opts && typeof opts.marginTop === 'number') ? opts.marginTop : 40;
  // default extraOffset increased to 18px so the symbol and its lines are pushed
  // slightly further down from the artwork (adjustable via opts.extraOffset)
  const extraOffset = (opts && typeof opts.extraOffset === 'number') ? opts.extraOffset : 18;

  // Draw token symbol top-left (shifted down by marginTop)
  // leftShift allows nudging the left column (symbol + lines) down independently
  // from the percent block on the right. Default is 5px as requested.
  const leftShift = (opts && typeof opts.leftShift === 'number') ? opts.leftShift : 5;
  const baseSymbolY = 40 + marginTop + extraOffset;
  const symbolY = baseSymbolY + leftShift;
  ctx.save();
  // Draw small semi-transparent panel behind symbol for readability
  const panelW = 360;
  const panelH = 72;
  ctx.fillStyle = 'rgba(0,0,0,0.35)';
  roundRect(ctx, 28, symbolY - 44, panelW, panelH, 8, true, false);
  ctx.font = 'bold 48px Arial';
  ctx.fillStyle = '#FFFFFF';
  ctx.textAlign = 'left';
  ctx.shadowColor = 'rgba(0,0,0,0.6)';
  ctx.shadowBlur = 8;
  ctx.fillText((symbol || 'UNKNOWN').toUpperCase(), 40, symbolY);
  ctx.restore();

    // Draw first scanned info line (like: 1st scanned 215.8k by @user 4m ago)
    const firstMcapDisplay = typeof firstMcapRaw === 'string' ? firstMcapRaw : (firstNum !== null ? formatNumberWithSuffix(firstNum) : 'Unknown');
    const firstAgo = firstCalledAt ? timeAgoShort(firstCalledAt) : '';
    const firstLine = `1st scanned ${firstMcapDisplay} by ${firstCalledBy || 'Unknown'} ${firstAgo ? firstAgo : ''}`;
    ctx.font = '28px Arial';
    ctx.fillStyle = '#FFFFFF';
  const firstLineY = symbolY + 50 + extraOffset;
    ctx.fillText(firstLine, 40, firstLineY);

    // Draw latest Mcap under it
    const latestMcapDisplay = typeof latestMcapRaw === 'string' ? latestMcapRaw : (latestNum !== null ? formatNumberWithSuffix(latestNum) : 'Unknown');
    const latestAgo = latestCalledAt ? timeAgoShort(latestCalledAt) : '';
    const latestLine = `Latest: ${latestMcapDisplay} ${latestAgo ? latestAgo : ''}`;
  const latestLineY = firstLineY + 40 + extraOffset;
    ctx.fillText(latestLine, 40, latestLineY);

  // Draw big percent on right (green if positive, red if negative) with stroke and shadow
  const pctText = pct === null ? 'N/A' : `${pct >= 0 ? '+' : ''}${pct.toFixed(0)}%`;
  ctx.font = 'bold 120px Arial';
  ctx.textAlign = 'center';
  ctx.shadowColor = 'rgba(0,0,0,0.7)';
  ctx.shadowBlur = 18;
  ctx.fillStyle = pct === null ? '#CCCCCC' : (pct >= 0 ? '#4CAF50' : '#F44336');

  // Layout safety: ensure percent + ratio never overlap the bottom of the canvas.
  // bottomMargin leaves space from the bottom edge; ratioOffset is vertical gap
  // between percent text and the small ratio display.
  const bottomMargin = (opts && typeof opts.bottomMargin === 'number') ? opts.bottomMargin : 80;
  const ratioOffset = 60;

  // Desired vertical position based on the original (pre-shift) left-side lines
  // so moving the left column won't move the percent block. Compute base
  // positions (without leftShift) and use those for percent layout.
  const baseFirstLineY = baseSymbolY + 50 + extraOffset;
  const baseLatestLineY = baseFirstLineY + 40 + extraOffset;
  const desiredPctY = baseLatestLineY + 150 + extraOffset;
  // Don't place percent above the (pre-shift) symbol area
  const minPctY = baseSymbolY + 120;
  // Ensure percent + ratio fit above bottomMargin
  const maxPctY = height - bottomMargin - ratioOffset - 10;

  // Clamp into safe band: at least minPctY but no lower than maxPctY
  let pctY = Math.min(Math.max(desiredPctY, minPctY), maxPctY);

  // Stroke for contrast
  ctx.lineWidth = 8;
  ctx.strokeStyle = 'rgba(0,0,0,0.6)';
  ctx.strokeText(pctText, 740, pctY);
  ctx.fillText(pctText, 740, pctY);

  // Draw small ratio (e.g., 1.37X) under percent if possible, positioned safely
  const ratioY = pctY + ratioOffset;
  if (firstNum !== null && latestNum !== null && firstNum > 0) {
    const ratio = (latestNum / firstNum).toFixed(2) + 'X';
    ctx.font = 'bold 36px Arial';
    ctx.fillStyle = '#FFFFFF';
    ctx.fillText(ratio, 740, ratioY);
  }

  // Branding
  ctx.font = '18px Arial';
  ctx.fillStyle = '#FFFFFF';
  ctx.textAlign = 'left';
  ctx.fillText('Powered by RadarAnalyzer.io', 40, height - 30);

    return canvas.toBuffer('image/png');
  } catch (err) {
    console.error('generatePnlCard error:', err.message);
    return null;
  }
}

// Short human-readable relative time (e.g., '4m ago', '2h ago') given UNIX seconds or Date
function timeAgoShort(ts) {
  try {
    let t = ts;
    if (typeof t === 'string') t = parseInt(t, 10);
    if (t > 1e12) t = Math.floor(t / 1000); // ms to s
    const now = Math.floor(Date.now() / 1000);
    const diff = now - (t || now);
    if (diff < 60) return `${diff}s ago`;
    if (diff < 3600) return `${Math.floor(diff / 60)}m ago`;
    if (diff < 86400) return `${Math.floor(diff / 3600)}h ago`;
    return `${Math.floor(diff / 86400)}d ago`;
  } catch (e) {
    return '';
  }
}

// Export the new function
module.exports.generateResultImage = generateResultImage;
module.exports.generatePnlCard = generatePnlCard;

// Helper: draw rounded rectangle
function roundRect(ctx, x, y, width, height, radius, fill, stroke) {
  if (typeof stroke === 'undefined') stroke = true;
  if (typeof radius === 'undefined') radius = 5;
  if (typeof radius === 'number') {
    radius = { tl: radius, tr: radius, br: radius, bl: radius };
  } else {
    var defaultRadius = { tl: 0, tr: 0, br: 0, bl: 0 };
    for (var side in defaultRadius) {
      radius[side] = radius[side] || defaultRadius[side];
    }
  }
  ctx.beginPath();
  ctx.moveTo(x + radius.tl, y);
  ctx.lineTo(x + width - radius.tr, y);
  ctx.quadraticCurveTo(x + width, y, x + width, y + radius.tr);
  ctx.lineTo(x + width, y + height - radius.br);
  ctx.quadraticCurveTo(x + width, y + height, x + width - radius.br, y + height);
  ctx.lineTo(x + radius.bl, y + height);
  ctx.quadraticCurveTo(x, y + height, x, y + height - radius.bl);
  ctx.lineTo(x, y + radius.tl);
  ctx.quadraticCurveTo(x, y, x + radius.tl, y);
  ctx.closePath();
  if (fill) ctx.fill();
  if (stroke) ctx.stroke();
}
